
********************************************************************************
********************************************************************************
* Date: November 2025
** Do: Code to reproduce all Country-sheets of World Inequality Report 2026
*** Author: Daniel Sanchez-Ordonez
********************************************************************************
********************************************************************************

/*
For information on how to use the widcode and the World Inequality Database, visit:
	*1. https://wid.world/codes-dictionary/
	*2. https://github.com/world-inequality-database/wid-stata-tool
	*3. https://github.com/world-inequality-database/wid-r-tool
	*4. https://wid.world/document/distributional-national-accounts-dina-guidelines-2025-methods-and-concepts-used-in-the-world-inequality-database/
	
	For other data inputs visit:
	*1. https://github.com/world-inequality-database/wid-world/tree/master/data-input
	
*/


****************************************************
* FINAL: All countries (aligned with R master list)
****************************************************

* Load full WID file (adjust path if needed)
use $raw/wid2024, clear
keep if widcode == "sptinc999j" & inlist(p, "p0p50", "p90p100") //992 before

* Define ISO list — alphabetically ordered, 51 countries (added ET for Ethiopia)
local countries "DZ AR AU BD BR CA CL CN CO CD DK EG ET FR DE HU IN ID IR IT CI JP KE ML MX MA MM NL NZ NE NG NO PK PH PL RU RW SA ZA KR ES SD SE TW TH TR AE GB US VN"

* Keep only those countries
gen keepme = 0
foreach c of local countries {
    replace keepme = 1 if iso == "`c'"
}
keep if keepme
drop keepme
drop currency
replace value = round(value, .001)
reshape wide value, i(year iso) j(p) string
renvars value*, pred(5)
save "$output/a1f1_small_top10bottom50_allcountries.dta", replace

****************************************************
* Export each country to its own Excel sheet
****************************************************
clear all
set more off

use "$output/a1f1_small_top10bottom50_allcountries.dta", clear

* Convert to percent
gen top10    = p90p100 * 100
gen bottom50 = p0p50   * 100

* Matching readable country names (same order, added Ethiopia)
local names "Algeria Argentina Australia Bangladesh Brazil Canada Chile China Colombia Congo Denmark Egypt Ethiopia France Germany Hungary India Indonesia Iran Italy IvoryCoast Japan Kenya Mali Mexico Morocco Myanmar Netherlands NewZealand Niger Nigeria Norway Pakistan Philippines Poland Russia Rwanda SaudiArabia SouthAfrica SouthKorea Spain Sudan Sweden Taiwan Thailand Turkey UnitedArabEmirates UK USA Vietnam"

* Output Excel path
local outfile "$output/a1f1_allcountries.xlsx"
capture erase "`outfile'"

drop if iso == "TR" & year < 1980

* Loop over both lists
local n : word count `countries'

forvalues i = 1/`n' {
    local code : word `i' of `countries'
    local cname : word `i' of `names'

    di as text "📊 Exporting `cname' (`code')..."

    preserve
        keep if iso == "`code'"
        keep year top10 bottom50
        order year top10 bottom50
        export excel using "`outfile'", sheet("`cname'_DATA", replace) firstrow(variables)
    restore
}

display as text "✅ All 50 countries exported successfully to:"
display as result "`outfile'"

The file which is outputted is left alone in dropbox, then with vba inside the _forgraphs file i create the graphs


*generates a1f1_allcountries.xlsx
*then open a1f1_allcountries.xlsx
*then alt+f11
*import file in Visual basic
*double click on export_f1_to_excel.bas
*alt + f8
*run
*provides final output 








****************************************************
* Create A1T1 small overview dataset (all countries)
****************************************************

clear all
set more off

cd "$raw-data"

use $wid2024.dta, clear


* === Keep only necessary WID codes (added xlceup999i for PPP conversion) ===
keep if inlist(widcode, ///
    "aptinc999j", /// avg income
    "sptinc999j", /// income share
    "ahweal999i", /// avg wealth
    "shweal999j", /// wealth share
    "spllin992f", /// female labor income share
    "iquali999i", /// transparency index
    "xlceup999i", /// PPP conversion rate
    "npopul999i")     /// total population

	

	
* === Keep relevant years per variable ===
keep if (inlist(year, 2014, 2024) & inlist(widcode, "aptinc999j")) ///
    | (year == 2024 & inlist(widcode, "sptinc999j", "ahweal999i", "npopul999i")) ///
    | (inlist(widcode, "iquali999i")) ///
    | (inlist(year, 2024) & widcode == "xlceup999i") ///
	| (inlist(year, 2023) & widcode == "shweal999j") 



* === Keep only relevant percentiles ===
keep if inlist(p, "p0p50", "p50p90", "p90p100", "p99p100", "p0p100")

preserve
	use $wid2024.dta, clear
	keep if inlist(widcode, "spllin992f")    
	keep if (inlist(year, 2014, 2024))
	keep if inlist(p,  "p0p100")
	tempfile female
	save `female'
restore

append using `female'


* === Keep only target countries ===
local countries "DZ AR AU BD BR CA CL CN CO CD DK EG ET FR DE HU IN ID IR IT CI JP KE ML MX MA MM NL NZ NE NG NO PK PH PL RU RW SA ZA KR ES SD SE TW TH TR AE GB US VN"
gen keepme = 0
foreach c of local countries {
    replace keepme = 1 if iso == "`c'"
}
keep if keepme
drop keepme
drop currency

replace year = 2024 if year == 2023
replace value = round(value, .001)

* === Reshape wide so each widcode becomes its own column ===
reshape wide value, i(iso year p) j(widcode) string
renvars value*, predrop(5)

* === Convert avg income and wealth to PPP euros using xlceup999i ===


* === Save dataset ===
save "$output/a1t1_small_allcountries.dta", replace


clear all

use "$output/a1t1_small_allcountries.dta", clear

bys iso (xlceup999i): replace xlceup999i = xlceup999i[1]
replace aptinc999j = aptinc999j / xlceup999i
replace ahweal999i = ahweal999i / xlceup999i
drop xlceup999i

* ------------------------------------------------------------
* 1. Compute Top 10 to Bottom 50 Income Gap  (for 2015 & 2025)
* ------------------------------------------------------------
preserve
    keep if inlist(year, 2014, 2024)
    gen b50 = aptinc999j if p == "p0p50"
    bys iso year (b50): replace b50 = b50[1]
    gen t10 = aptinc999j if p == "p90p100"
    bys iso year (t10): replace t10 = t10[1]
    gen t10b50 = t10 / b50
    replace t10b50 = . if p != "p0p100"
    tempfile ratio
    save `ratio'
restore
merge 1:1 iso year p using `ratio', nogenerate

*display c(tmpdir)

* ------------------------------------------------------------
* 2. Keep only the variables relevant to the table
* ------------------------------------------------------------
keep iso year p aptinc999j sptinc999j ahweal999i shweal999j spllin992f iquali999i t10b50 npopul999i
order iso year p aptinc999j sptinc999j ahweal999i shweal999j spllin992f t10b50 iquali999i npopul999i

* ------------------------------------------------------------
* 3. Filter years & percentiles we actually need
* ------------------------------------------------------------
* 2025: for all income/wealth levels + shares
* 2015 & 2025: for female labor share and income ratio
* All available years for transparency

keep if (inlist(year, 2024) & inlist(p,"p0p100","p0p50","p50p90","p90p100","p99p100")) ///
    | (inlist(year,2014,2024) & p=="p0p100" & !missing(t10b50)) ///
    | (!missing(iquali999i))


* ------------------------------------------------------------
* 4. Prepare and label data for Table 1
* ------------------------------------------------------------

* We only need one record per country–year–percentile
duplicates drop iso year p, force

* Label variable names to readable equivalents
label var aptinc999j   "Average Income (PPP €)"
label var sptinc999j   "Income Share (%)"
label var ahweal999i   "Average Wealth (PPP €)"
label var shweal999j   "Wealth Share (%)"
label var spllin992f   "Female Labor Income Share (%)"
label var iquali999i   "Transparency Index"
label var t10b50       "Top 10% to Bottom 50% Income Gap"
label var npopul999i   "Total Pop. 2025"

* ------------------------------------------------------------
* 5. Restrict to relevant percentiles for table display
* ------------------------------------------------------------
keep if inlist(p,"p0p100","p0p50","p50p90","p90p100","p99p100")

* ------------------------------------------------------------
* 6. Generate clear labels for each percentile group
* ------------------------------------------------------------
gen group = ""
replace group = "Full pop."   if p=="p0p100"
replace group = "Bottom 50%"  if p=="p0p50"
replace group = "Middle 40%"  if p=="p50p90"
replace group = "Top 10%"     if p=="p90p100"
replace group = "Top 1%"      if p=="p99p100"


* ------------------------------------------------------------
* 7. Add full country names
* ------------------------------------------------------------
gen country_name = ""

replace country_name = "Algeria"          if iso=="DZ"
replace country_name = "Argentina"        if iso=="AR"
replace country_name = "Australia"        if iso=="AU"
replace country_name = "Bangladesh"       if iso=="BD"
replace country_name = "Brazil"           if iso=="BR"
replace country_name = "Canada"           if iso=="CA"
replace country_name = "Chile"            if iso=="CL"
replace country_name = "China"            if iso=="CN"
replace country_name = "Colombia"         if iso=="CO"
replace country_name = "Congo, DR"        if iso=="CD"
replace country_name = "Denmark"          if iso=="DK"
replace country_name = "Egypt"            if iso=="EG"
replace country_name = "Ethiopia"         if iso=="ET"
replace country_name = "France"           if iso=="FR"
replace country_name = "Germany"          if iso=="DE"
replace country_name = "Hungary"          if iso=="HU"
replace country_name = "India"            if iso=="IN"
replace country_name = "Indonesia"        if iso=="ID"
replace country_name = "Iran"             if iso=="IR"
replace country_name = "Italy"            if iso=="IT"
replace country_name = "Ivory Coast"      if iso=="CI"
replace country_name = "Japan"            if iso=="JP"
replace country_name = "Kenya"            if iso=="KE"
replace country_name = "Mali"             if iso=="ML"
replace country_name = "Mexico"           if iso=="MX"
replace country_name = "Morocco"          if iso=="MA"
replace country_name = "Myanmar"          if iso=="MM"
replace country_name = "Netherlands"      if iso=="NL"
replace country_name = "New Zealand"      if iso=="NZ"
replace country_name = "Niger"            if iso=="NE"
replace country_name = "Nigeria"          if iso=="NG"
replace country_name = "Norway"           if iso=="NO"
replace country_name = "Pakistan"         if iso=="PK"
replace country_name = "Philippines"      if iso=="PH"
replace country_name = "Poland"           if iso=="PL"
replace country_name = "Russia"           if iso=="RU"
replace country_name = "Rwanda"           if iso=="RW"
replace country_name = "Saudi Arabia"     if iso=="SA"
replace country_name = "South Africa"     if iso=="ZA"
replace country_name = "South Korea"      if iso=="KR"
replace country_name = "Spain"            if iso=="ES"
replace country_name = "Sudan"            if iso=="SD"
replace country_name = "Sweden"           if iso=="SE"
replace country_name = "Taiwan"           if iso=="TW"
replace country_name = "Thailand"         if iso=="TH"
replace country_name = "Turkey"           if iso=="TR"
replace country_name = "United Arab Emirates" if iso=="AE"
replace country_name = "United Kingdom"   if iso=="GB"
replace country_name = "United States"    if iso=="US"
replace country_name = "Vietnam"          if iso=="VN"

order country_name iso year group p aptinc999j sptinc999j ahweal999i shweal999j spllin992f  t10b50 iquali999i npopul999i //female share between sweal and t10b50



* ------------------------------------------------------------
* 7. Sort cleanly and save
* ------------------------------------------------------------
sort iso year p
save $output/a1t1_small_allcountries_clean.dta, replace

display as text "✅ Clean Table 1 data saved as: $output/a1t1_small_allcountries_clean.dta"

****************************************************
* Export all Table 1 data to Excel — one sheet per country
****************************************************

use "$output/a1t1_small_allcountries_clean.dta", clear

* get full-pop 2024 ahweal per country
bysort iso year: egen base_ahweal = max(cond(group=="Full pop." & p=="p0p100", ahweal999i, .))

* bottom 50 from that base
replace ahweal999i = base_ahweal * shweal999j if year == 2024 & shweal999j < .

* then multipliers
replace ahweal999i = ahweal999i*100    if year == 2024 & p == "p99p100"
replace ahweal999i = ahweal999i*10     if year == 2024 & p == "p90p100"
replace ahweal999i = ahweal999i*(10/4) if year == 2024 & p == "p50p90"



* Define country ISO and name lists (same as before)
local countries "DZ AR AU BD BR CA CL CN CO CD DK EG ET FR DE HU IN ID IR IT CI JP KE ML MX MA MM NL NZ NE NG NO PK PH PL RU RW SA ZA KR ES SD SE TW TH TR AE GB US VN"
local names "Algeria Argentina Australia Bangladesh Brazil Canada Chile China Colombia Congo Denmark Egypt Ethiopia France Germany Hungary India Indonesia Iran Italy IvoryCoast Japan Kenya Mali Mexico Morocco Myanmar Netherlands NewZealand Niger Nigeria Norway Pakistan Philippines Poland Russia Rwanda SaudiArabia SouthAfrica SouthKorea Spain Sudan Sweden Taiwan Thailand Turkey UnitedArabEmirates UK USA Vietnam"

* Output Excel file path
local outfile "$output/a1t1_allcountries.xlsx"
capture erase "`outfile'"

* Count how many countries
local n : word count `countries'

forvalues i = 1/`n' {
    local code : word `i' of `countries'
    local cname : word `i' of `names'

    di as text "📊 Exporting `cname' (`code')..."

    preserve
        keep if iso == "`code'"
        keep country_name iso year group p aptinc999j sptinc999j ahweal999i shweal999j spllin992f  t10b50 iquali999i npopul999i
        order country_name iso year group p aptinc999j sptinc999j ahweal999i shweal999j spllin992f t10b50 iquali999i npopul999i
        export excel using "`outfile'", sheet("`cname'_DATA", replace) firstrow(variables)
    restore
}

display as text "✅ All countries exported successfully to:"
display as result "`outfile'"


*generates a1t1_allcountries.xlsx
*then open a1t1_allcountries.xlsx
*then alt+f11
*import file in Visual basic
*double click on export_t1_to_excel.bas
*alt + f8
*run
*provides final output 
