********************************************************************************
********************************************************************************
* Date: November 2025
** Do: Code to reproduce Chapter 8 of World Inequality Report 2026
*** Author: Ricardo Gómez-Carrera
	* 1. Figure 8.1. Working class representation has always been low and has further deteriorated in recent decades
	* 2. Figure 8.2. Educated voters increasingly support the left, while high-income voters continue leaning right
	* 3. Figure 8.3. The reversal of educational divides in Western democracies
	* 4. Figure 8.4. The stability/decline of income divides in Western democracies
	* 5. Figure 8.5. The fragmentation of political cleavage structures in Western democracies
	* 6. Figure 8.6. Income and educational divides in non-Western democracies
	* 7. Figure 8.7. Rise of tripartition and income in France
	* 8. Figure 8.8. The territorial divide (urban vs. rural) in France
	* 9. Figure 8.9. Geosocial class explanatory power is stronger than ever in Francce

********************************************************************************
********************************************************************************

/*
For information on how to use the widcode and the World Inequality Database, visit:
	*1. https://wid.world/codes-dictionary/
	*2. https://github.com/world-inequality-database/wid-stata-tool
	*3. https://github.com/world-inequality-database/wid-r-tool
	*4. https://wid.world/document/distributional-national-accounts-dina-guidelines-2025-methods-and-concepts-used-in-the-world-inequality-database/
	
	For other data inputs visit:
	*1. https://github.com/world-inequality-database/wid-world/tree/master/data-input
	
*/


**# Figure 8.1. Working class representation has always been low and has further deteriorated in recent decades
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Cagé (2024)
Figure 11 in Cagé (2024)

Paper
https://wid.world/document/political-inequality-wid-world-working-paper-2023-22/

Data provided by Julia Cagé
*/


*** France
import excel "$raw/2018_05_03_CAHIERSFRANCAIS_rouban_em.xlsx", sheet("julia") cellrange(A2) firstrow clear
rename A year
drop if year==.
gen pays="France" 
rename year annee 
order pays annee
rename Employésetouvriersduprivé_pc pourcentage_ouvriers_employes
keep pays annee pourcentage_ouvriers_employes
	preserve 
	import excel "$raw/EmployesOuvriers_2022.xlsx", sheet("data") firstrow clear
	tempfile France2022
	save `France2022', replace 
restore 
append using `France2022'	
tempfile France 
save `France', replace 

*** UK 
import excel "$raw/MPs_csp.xlsx", sheet("julia") firstrow clear
	reshape long y, i(CSP) j(year)
	gen party="all"
	tempfile all
	save `all', replace
import excel "$raw/MPs_csp.xlsx", sheet("Labour") firstrow clear
	reshape long y, i(CSP) j(year)
	gen party="Labour"
	tempfile Labour
	save `Labour', replace
import excel "$raw/MPs_csp.xlsx", sheet("Conservative") firstrow clear
	reshape long y, i(CSP) j(year)
	gen party="Conservative"
	append using `all'
	append using `Labour'
replace CSP="ArmedServices" if CSP=="Armed services"
replace CSP="Publisher" if CSP=="Publisher/journalist"
replace CSP="CivilService" if CSP=="Civil service/local government"
replace CSP="Executive" if CSP=="Company Executive/Director"
replace CSP="MisWC" if CSP=="Miscellaneous white collar"
replace CSP="OManual" if CSP=="Other Manual"
replace CSP="OProfessions" if CSP=="Other Professions"
replace CSP="OBusiness" if CSP=="Other business"
replace CSP="Politician" if CSP=="Politician/political organiser"

reshape wide y, i(year party) j(CSP) string

gen tot_manual=yMiner+yOManual
gen sh_manual=(tot_manual/yTotal)*100

rename party parti 
replace parti="ensemble" if parti=="all" 
gen pays="Royaume-Uni" 
rename year annee 
order pays annee parti
rename sh_manual pourcentage_ouvriers_employes
keep  pays annee parti pourcentage_ouvriers_employes
keep if parti=="ensemble"

preserve 
	import excel "$raw/CSP_2017_2019.xlsx", sheet("data") firstrow clear
	tempfile UK2017_2019
	save `UK2017_2019', replace 
restore 	
append using `UK2017_2019'
tempfile UK
save `UK', replace 

*** US 
import excel "$raw/Cash Ceiling Figure 1.2.xlsx", sheet("julia") firstrow clear	
replace var="congress_carnes" if var=="congress (carnes)"
replace var="congress_roster" if var=="congress (roster)"
replace var="state_leg" if var=="state leg"
replace var="women_congress" if var=="women in congress (roster + update)"
replace var="women_state_leg" if var=="women in state leg"

gen n=_n
reshape long y, i(n) j(annee) string 
destring annee, replace 
drop n
levelsof var, local(levels) 
	foreach l of local levels {
	preserve 
		keep if var == "`l'"
		rename y `l'
		tempfile `l'
		save ``l'', replace 
	restore 
	}
clear 
use `congress_carnes', replace 
merge 1:1 annee using `congress_roster', nogen
merge 1:1 annee using  `state_leg', nogen
merge 1:1 annee using `women_congress', nogen
merge 1:1 annee using `women_state_leg', nogen
drop var
gen pourcentage_ouvriers_employes=congress_carnes
	replace pourcentage_ouvriers_employes=congress_roster if congress_roster!=. & congress_carnes==.
tab annee if pourcentage_ouvriers_employes!=.
replace pourcentage_ouvriers_employes=pourcentage_ouvriers_employes*100
gen pays="US"
order pays 
tempfile US
save `US', replace 	

*** All
use `France', replace 
append using `UK'
append using `US'

twoway (connected pourcentage_ouvriers_employes annee if pays=="France", sort msize(small))	 ///	/* 1958-2024 */
	(connected pourcentage_ouvriers_employes annee if pays=="Royaume-Uni", sort lpattern(dash) msymbol(triangle) msize(small)) ///	/* 1951-2023 */
	(connected pourcentage_ouvriers_employes annee if pays=="US", sort lpattern(dash_dot) msymbol(square) msize(small)),		///		/* 1901-2007 */
	xlabel(1900(5)2025, labsize(small) angle(45)) xtitle("")	///
	ytitle("Share of working-class MPs (%)") ylabel(0(4)20, angle(horizontal) format(%15.0fc)) ///
	legend(row(1) order(1 "France" 2 "United Kingdom" 3 "United States") size(small) position(6)) ///
	graphregion(color(white))





**# Figure 8.2. Educated voters increasingly support the left, while high-income voters continue leaning right
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from World Political Cleavages And Inequality Database (https://wpid.world/resources/)
Figure I in Gethin et al. (2021) 

Paper
https://academic.oup.com/qje/article/137/1/1/6383014

Replication Package
https://wpid.world/resources/en/zip/wpid-macro.zip


Also, Chapter1.xlsx, Figure 1.1 in
https://wpid.world/resources/en/zip/GethinMartinezPiketty2021ENFiguresTables_full.zip

Updated data from https://explore.wpid.world/intime

*/

import excel "$raw/WPID-graphdatas.xlsx", sheet("gr1_inc_dc90") clear firstrow

keep if inlist(countryName, "Australia","United Kingdom","Canada","Denmark","France","Germany","Italy") | ///
inlist(countryName,"Netherlands","Norway","Sweden","Switzerland","United States")
keep country year value
destring year, replace

* Keep only the requested years
keep if inrange(yearLabel, 1955, 2025)

* Build 5-year bins anchored at 1955
gen binstart = floor((yearLabel - 1955)/5)*5 + 1955
gen binend   = binstart + 4

* Period label like "2000-04", "1955-59"
gen str9 period = string(binstart) + "-" + substr(string(binend, "%04.0f"), 3, 2)
label var period "5-year period"

replace binstart=2020 if binstart==2025
replace period="2020-24" if period=="2025-29" // change the only observation for 2025-29 for it to be 2020-24

* Average value within each country x period
collapse (mean) value, by(binstart period)
ren value gr1_inc_dc90
tempfile gr1_inc_dc90
save `gr1_inc_dc90',replace



import excel "$raw/WPID-graphdatas.xlsx", sheet("gr1_educ_dc90") clear firstrow

keep if inlist(countryName, "Australia","United Kingdom","Canada","Denmark","France","Germany","Italy") | ///
inlist(countryName,"Netherlands","Norway","Sweden","Switzerland","United States")
keep country year value
destring year, replace

* Keep only the requested years
keep if inrange(yearLabel, 1955, 2025)

* Build 5-year bins anchored at 1955
gen binstart = floor((yearLabel - 1955)/5)*5 + 1955
gen binend   = binstart + 4

* Period label like "2000-04", "1955-59"
gen str9 period = string(binstart) + "-" + substr(string(binend, "%04.0f"), 3, 2)
label var period "5-year period"

replace binstart=2020 if binstart==2025
replace period="2020-24" if period=="2025-29" // change the only observation for 2025-29 for it to be 2020-24
* Average value within each country x period
collapse (mean) value, by(binstart period)
ren value gr1_educ_dc90
merge bin period using `gr1_inc_dc90'
drop _merge
sort period
replace period="2020-25" if period=="2020-24"
label var gr1_educ_dc90 "Difference between (% of top 10% educated voting left) and (% of bottom 90% educated voting left)"
label var gr1_inc_dc90 "Difference between (% of top 10% earners voting left) and (% of bottom 90% earners voting left)"
tw (line gr1_educ_dc90  binstart) (line  gr1_inc_dc90 binstart), legend(pos(6))
export excel using "$outputs/WIR_2026_Chapter8.xlsx", sheet("data-F8.2") sheetmod cell(A2) first(varl) 



**# Figures 8.3. and 8.4.
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from World Political Cleavages And Inequality Database (https://wpid.world/resources/)
Figures II and III in Gethin et al. (2021) 

Paper
https://academic.oup.com/qje/article/137/1/1/6383014

Replication Package
https://wpid.world/resources/en/zip/wpid-macro.zip


Also, Chapter1.xlsx, Figures 1.2 and 1.3 in
https://wpid.world/resources/en/zip/GethinMartinezPiketty2021ENFiguresTables_full.zip

Updated data from https://explore.wpid.world/intime

*/


*Panel a
import excel "$raw/WPID-graphdatas.xlsx", sheet("gr1_educ_dc90") clear firstrow
keep if inlist(countryName, "Australia","United Kingdom","Canada","Denmark","Finland","Iceland","Ireland") | ///
inlist(countryName,"New Zealand","Norway","Sweden","United States")
keep country year value
destring year, replace
drop if year<1950

* Build decade bins anchored at 1950
gen binstart = floor((year - 1950)/10)*10 + 1950
gen binend   = binstart + 9

* Period label like "1950-59", "2000-09"
gen str9 period = string(binstart) + "-" + substr(string(binend, "%04.0f"), 3, 2)
label var period "10-year period"

collapse (mean) value, by(countryName period)

preserve
collapse (mean) value, by( period)
g countryName="Average"
tempfile average
save  `average', replace
restore

append using `average'

greshape wide value, i(period) j(countryName) string
renvars value*, predrop(5)


label var Australia "Australia"
label var Average "Average"
label var Canada "Canada"
label var Denmark "Denmark"
label var Finland "Finland"
label var Iceland "Iceland"
label var Ireland "Ireland"
label var New_Zealand "New Zealand"
label var Norway "Norway"
label var Sweden "Sweden"
label var United_Kingdom "Britain" 
label var United_States "United States"
order period Australia United_Kingdom Canada Denmark Finland Iceland Ireland New_Zealand Norway Sweden  United_States Average 
replace period ="2020-25" if period=="2020-29"
export excel using "$outputs/WIR_2026_Chapter8.xlsx", sheet("data-F8.3.a.") sheetmod cell(A2) first(varl) 


*Panel b
import excel "$raw/WPID-graphdatas.xlsx", sheet("gr1_educ_dc90") clear firstrow
drop if inlist(countryName, "Australia","United Kingdom","Canada","Denmark","Finland","Iceland","Ireland") | ///
inlist(countryName,"New Zealand","Norway","Sweden","United States")
keep country year value
destring year, replace
drop if year<1950
keep if value!=.

* Build decade bins anchored at 1950
gen binstart = floor((year - 1950)/10)*10 + 1950
gen binend   = binstart + 9

* Period label like "1950-59", "2000-09"
gen str9 period = string(binstart) + "-" + substr(string(binend, "%04.0f"), 3, 2)
label var period "10-year period"

collapse (mean) value, by(countryName period)

preserve
collapse (mean) value, by( period)
g countryName="Average"
tempfile average
save  `average', replace
restore

append using `average'

greshape wide value, i(period) j(countryName) string
renvars value*, predrop(5)


label var Austria "Austria"
label var Average "Average" 
label var Belgium "Belgium"
label var France "France"
label var Germany "Germany"
label var Italy "Italy"
label var Luxembourg "Luxembourg" 
label var Netherlands "Netherlands" 
label var Portugal "Portugal" 
label var Spain "Spain" 
label var Switzerland "Switzerland"

order period Austria Average Belgium France Germany Italy Luxembourg Netherlands Portugal Spain Switzerland Average 
replace period ="2020-25" if period=="2020-29"
export excel using "$outputs/WIR_2026_Chapter8.xlsx", sheet("data-F8.3.b.") sheetmod cell(A2) first(varl) 




*Panel c
import excel "$raw/WPID-graphdatas.xlsx", sheet("gr1_inc_dc90") clear firstrow
keep if inlist(countryName, "Australia","United Kingdom","Canada","Denmark","Finland","Iceland","Ireland") | ///
inlist(countryName,"New Zealand","Norway","Sweden","United States")
keep country year value
destring year, replace
drop if year<1950

* Build decade bins anchored at 1950
gen binstart = floor((year - 1950)/10)*10 + 1950
gen binend   = binstart + 9

* Period label like "1950-59", "2000-09"
gen str9 period = string(binstart) + "-" + substr(string(binend, "%04.0f"), 3, 2)
label var period "10-year period"

collapse (mean) value, by(countryName period)

preserve
collapse (mean) value, by( period)
g countryName="Average"
tempfile average
save  `average', replace
restore

append using `average'

greshape wide value, i(period) j(countryName) string
renvars value*, predrop(5)


label var Australia "Australia"
label var Average "Average"
label var Canada "Canada"
label var Denmark "Denmark"
label var Finland "Finland"
label var Iceland "Iceland"
label var Ireland "Ireland"
label var New_Zealand "New Zealand"
label var Norway "Norway"
label var Sweden "Sweden"
label var United_Kingdom "Britain" 
label var United_States "United States"
order period Australia United_Kingdom Canada Denmark Finland Iceland Ireland New_Zealand Norway Sweden  United_States Average 
replace period ="2020-25" if period=="2020-29"
export excel using "$outputs/WIR_2026_Chapter8.xlsx", sheet("data-F8.4.a.") sheetmod cell(A2) first(varl) 


*Panel d
import excel "$raw/WPID-graphdatas.xlsx", sheet("gr1_inc_dc90") clear firstrow
drop if inlist(countryName, "Australia","United Kingdom","Canada","Denmark","Finland","Iceland","Ireland") | ///
inlist(countryName,"New Zealand","Norway","Sweden","United States")
keep country year value
destring year, replace
drop if year<1950
keep if value!=.

* Build decade bins anchored at 1950
gen binstart = floor((year - 1950)/10)*10 + 1950
gen binend   = binstart + 9

* Period label like "1950-59", "2000-09"
gen str9 period = string(binstart) + "-" + substr(string(binend, "%04.0f"), 3, 2)
label var period "10-year period"

collapse (mean) value, by(countryName period)

preserve
collapse (mean) value, by( period)
g countryName="Average"
tempfile average
save  `average', replace
restore

append using `average'

greshape wide value, i(period) j(countryName) string
renvars value*, predrop(5)


label var Austria "Austria"
label var Average "Average" 
label var Belgium "Belgium"
label var France "France"
label var Germany "Germany"
label var Italy "Italy"
label var Luxembourg "Luxembourg" 
label var Netherlands "Netherlands" 
label var Portugal "Portugal" 
label var Spain "Spain" 
label var Switzerland "Switzerland"

order period Austria Average Belgium France Germany Italy Luxembourg Netherlands Portugal Spain Switzerland Average 
replace period ="2020-25" if period=="2020-29"
export excel using "$outputs/WIR_2026_Chapter8.xlsx", sheet("data-F8.3.b.") sheetmod cell(A2) first(varl) 


**# Figure 8.5. The fragmentation of political cleavage structures in Western democracies
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from World Political Cleavages And Inequality Database (https://wpid.world/resources/)
Figure VI in Gethin et al. (2021) 

Paper
https://academic.oup.com/qje/article/137/1/1/6383014

Replication Package
https://wpid.world/resources/en/zip/wpid-macro.zip


Also, Chapter1.xlsx, Figure 1.5 in
https://wpid.world/resources/en/zip/GethinMartinezPiketty2021ENFiguresTables_full.zip

Updated data from https://explore.wpid.world/intime
and provided by Amory Gethin

*/




**# Figure 8.6. Income and educational divides in non-Western democracies
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- *
/*
Data from World Political Cleavages And Inequality Database (https://wpid.world/resources/)


Paper
https://academic.oup.com/qje/article/137/1/1/6383014

Replication Package
https://wpid.world/resources/en/zip/wpid-macro.zip


Also, Chapter1.xlsx, Figure 1.7 in
https://wpid.world/resources/en/zip/GethinMartinezPiketty2021ENFiguresTables_full.zip

Updated data from https://explore.wpid.world/intime

*/

foreach var in educ inc{
import excel "$outputs/WPID-graphdatas.xlsx", sheet("vote_`var'_dp50") clear firstrow
keep if inlist(countryName, "Argentina","Brazil","Botswana","Chile","Colombia") | ///
    inlist(countryName,"Costa Rica","Czech Republic","Algeria","Ghana","Hong Kong") | ///
    inlist(countryName, "Hungary","Indonesia","Israel","India","Iraq") |  ///
    inlist(countryName,"Japan","South Korea","Mexico","Malaysia","Nigeria") | ///
    inlist(countryName, "Peru","Philippines","Pakistan","Poland","Senegal") |  ///
    inlist(countryName,"Thailand","Turkey","Taiwan","South Africa")

keep country year value
destring year, replace
drop if countryName=="Japan" & year>2013 //keep Japan 2013 (it has both education and income)
*drop if countryName=="Israel" & year>2013 //keep Israel 2013 (it has both education and income)

sort countryName yearLabel
by countryName: keep if _n == _N

ren value vote_`var'_dp50


tempfile vote_`var'_dp50
save  `vote_`var'_dp50', replace
}

use `vote_inc_dp50',clear
merge 1:1 country year using `vote_educ_dp50'
replace year =2013 if countryName=="Israel" //based on the Excel for the Book
replace vote_inc_dp50=	9.360590935	if countryName=="Israel" //based on the Excel for the Book Poliical Cleavages
replace vote_educ_dp50=3.467447996	if countryName=="Israel" //based on the Excel for the Book Poliical Cleavages
drop _merge
ren countryName isoname
ren yearLabel year 
replace isoname="Czechia"	if isoname=="Czech Republic" 
replace isoname="Turkiye" if isoname=="Turkey"
order iso year vote_inc_dp50
sum year

label variable isoname "isoname" 
label variable year "year" 
label variable vote_inc_dp50 "vote_inc_dp50" 
label variable vote_educ_dp50 "vote_educ_dp50"




preserve	
import excel using "$raw/country-codes-new.xlsx", firstrow clear
renvars code shortname / iso isoname
keep iso isoname 
replace isoname="Turkiye" if iso=="TR"
replace isoname="Czechia" if iso=="CZ"
replace isoname="Russia" if iso=="RU"
replace isoname="Eswatini" if iso=="SZ"
replace isoname="Syria" if iso=="SY"
replace isoname="South Korea" if iso=="KR"
tempfile isoname 
save `isoname', replace	
restore

merge 1:m isoname using `isoname'
keep if _merge==3
drop _merge
drop if iso=="QT"

g region=""
foreach reg in MENA RUCA SSAF SSEA EURO LATA NAOC EASA{
	foreach c in $`reg'{
		replace region="`reg'" if iso=="`c'"
	}	
}
keep if region!=""

replace region = "East Asia"                  if region == "EASA"
replace region = "Europe"                     if region == "EURO"
replace region = "Latin America"              if region == "LATA"
replace region = "MENA"                       if region == "MENA"
replace region = "North America & Oceania"    if region == "NAOC"
replace region = "South & Southeast Asia"     if region == "SSEA"
replace region = "Russia & Central Asia"      if region == "RUCA"
replace region = "Sub-Saharan Africa"         if region == "SSAF"

export excel using "$outputs/WIR_2026_Chapter8.xlsx", sheet("data-F8.6") sheetmod cell(A1) first(varl) 



**# Figure 8.7. Rise of tripartition and income in France
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Cagé and Piketty (2024).
Figure 11.27 in Cagé and Piketty (2024).

Paper
https://conflit-politique-data.ams3.cdn.digitaloceanspaces.com/pdf/CagePiketty2023Graphiques.pdf

Replication Package
https://conflit-politique-data.ams3.cdn.digitaloceanspaces.com/zip/CagePiketty2023Graphiques.zip
*/


**# Figure 8.8. The territorial divide (urban vs. rural) in France
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Cagé and Piketty (2024).
Figures 22 and 23 in Cagé and Piketty (2024).

Paper
https://wid.world/document/le-debut-de-la-fin-de-la-tripartition-elections-europeennes-et-inegalites-sociales-en-france-1994-2024-world-inequality-lab-working-paper-2024-18/

Replication Package
https://wid.world/document/data-file-to-le-debut-de-la-fin-de-la-tripartition-elections-europeennes-et-inegalites-sociales-en-france-1994-2024-world-inequality-lab-working-paper-2024-18/

*/



**# Figure 8.9. Geosocial class explanatory power is stronger than ever in Francce
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Cagé and Piketty (2024).
Figure 13.23 in Cagé and Piketty (2024).

Paper
https://wid.world/document/le-debut-de-la-fin-de-la-tripartition-elections-europeennes-et-inegalites-sociales-en-france-1994-2024-world-inequality-lab-working-paper-2024-18/
https://conflit-politique-data.ams3.cdn.digitaloceanspaces.com/pdf/CagePiketty2023Graphiques.pdf

Replication Package
https://conflit-politique-data.ams3.cdn.digitaloceanspaces.com/zip/CagePiketty2023Graphiques.zip
*/