********************************************************************************
********************************************************************************
* Date: November 2025
** Do: Code to reproduce Chapter 5 of World Inequality Report 2026
*** Author: Ricardo Gómez-Carrera 
/*
Based on the code provided by Gastón Nievas for the paper Nievas and Sodano (2025)
https://wid.world/document/has-the-us-exorbitant-privilege-become-a-rich-world-privilege-rates-of-return-and-foreign-assets-from-a-global-perspective-1970-2022-world-inequality-lab-working-paper-2024-14/
*/
	* 1. Figure 5.1. The U.S. exorbitant privilege has evolved into a structural privilege of the rich world
	* 2. Figure 5.2. Rich countries receive 1% of their GDP from the rest of the world due to financial privilege
	* 3. Figure 5.3. Privilege persists for the U.S. (and its region) despite negative net foreign asset positions
	* 4. Figure 5.4. There is a persistent net income transfer from poor to rich
	* 5. Figure 5.5. Rich countries are global financial rentiers by political design, not because of market dynamics
	* 6. Figure 5.6. Poor countries finance the privilege through asymmetric portfolios
	* 7. Figure 5.7. Poorer countries face lower asset returns and higher liability costs
	* 8. Figure 5.8. Poorer countries can spend less on public services, exacerbating inequality
	* 9. Figure 5.9. These structural asymmetries call for reforms in the international financial, trade, and monetary system
	*10. Figure Box 5.1. Exorbitant duty is not so exorbitant

********************************************************************************
********************************************************************************

/*
For information on how to use the widcode and the World Inequality Database, visit:
	*1. https://wid.world/codes-dictionary/
	*2. https://github.com/world-inequality-database/wid-stata-tool
	*3. https://github.com/world-inequality-database/wid-r-tool
	*4. https://wid.world/document/distributional-national-accounts-dina-guidelines-2025-methods-and-concepts-used-in-the-world-inequality-database/
	
	For other data inputs visit:
	*1. https://github.com/world-inequality-database/wid-world/tree/master/data-input
	
*/




**# Create Exorbitant Privilege dataset
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Create Exorbitant Privilege dataset          *
* ---------------------------------------------------------------------------- * 
clear 
insobs 1 
g use_widcode="$use_widcode"

if use_widcode=="YES"{
	wid, indicators(mgdpro mptfnx mptfon mptfop mptfor mptfpx mptfrp mptfrr mptfrx mfdinx) age(999) pop(i) clear 
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode,"mgdpro999i", "mptfnx999i", "mptfon999i", "mptfop999i", "mptfor999i") | ///
			inlist(widcode,"mptfpx999i", "mptfrp999i", "mptfrr999i","mptfrx999i", "mfdinx999i")	
}
else{
use if  inlist(widcode,"mgdpro999i", "mptfnx999i", "mptfon999i", "mptfop999i", "mptfor999i") | ///
			inlist(widcode,"mptfpx999i", "mptfrp999i", "mptfrr999i","mptfrx999i", "mfdinx999i")	& ///
			p == "p0p100" using $wid2024, clear
}	

tempfile nataccounts1 
save `nataccounts1'


clear 
insobs 1 
g use_widcode="$use_widcode"
if use_widcode=="YES" {
	wid, indicators(mfdipx mfdirx mfdimp mpinrx mpinpx mnninc mptfrn xlcusx inyixx npopul) age(999) pop(i) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "mfdipx999i", "mfdirx999i", "mfdimp999i", "mpinrx999i", "mpinpx999i") | ///
		inlist(widcode,"mnninc999i", "mptfrn999i", "xlcusx999i", "inyixx999i", "npopul999i")
}
else {
	use if inlist(widcode, "mfdipx999i", "mfdirx999i", "mfdimp999i", "mpinrx999i", "mpinpx999i") | ///
		inlist(widcode,"mnninc999i", "mptfrn999i", "xlcusx999i", "inyixx999i", "npopul999i") & ///
		p == "p0p100" using $wid2024, clear
}
tempfile nataccounts2 
save `nataccounts2'

clear 
insobs 1 
g use_widcode="$use_widcode"
if use_widcode=="YES" {
	wid, indicators(mptdrx mpterx mptrrx mptdxa mptexa mptrxa mptdpx mptepx mptdxd mptexd) age(999) pop(i) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "mptdrx999i", "mpterx999i", "mptrrx999i", "mptdxa999i", "mptexa999i") | ///
		inlist(widcode,"mptrxa999i", "mptdpx999i", "mptepx999i", "mptdxd999i", "mptexd999i")
}
else {
	use if inlist(widcode, "mptdrx999i", "mpterx999i", "mptrrx999i", "mptdxa999i", "mptexa999i") | ///
		inlist(widcode,"mptrxa999i", "mptdpx999i", "mptepx999i", "mptdxd999i", "mptexd999i") & ///
		p == "p0p100" using $wid2024, clear
}
tempfile nataccounts3 
save `nataccounts3'


clear 
insobs 1 
g use_widcode="$use_widcode"
if use_widcode=="YES" {
	wid, indicators(mnwgxa mnwgxd mfdixa mfdixd mptfxa mptfxd mtbnnx mtgnnx mtsnnx mtaxnx mscinx mfkanx mcomnx) age(999) pop(i) clear
	ren (variable country percentile) (widcode iso p)
	drop age pop
	g currency=""
	keep if inlist(widcode, "mnwgxa999i", "mnwgxd999i", "mfdixa999i", "mfdixd999i", "mptfxa999i", "mptfxd999i") | ///
		inlist(widcode,"mtbnnx999i", "mtgnnx999i", "mtsnnx999i", "mtaxnx999i", "mscinx999i", "mfkanx999i", "mcomnx999i")
}
else {
	use if inlist(widcode, "mnwgxa999i", "mnwgxd999i", "mfdixa999i", "mfdixd999i", "mptfxa999i", "mptfxd999i") | ///
		inlist(widcode,"mtbnnx999i", "mtgnnx999i", "mtsnnx999i", "mtaxnx999i", "mscinx999i", "mfkanx999i", "mcomnx999i") & ///
		p == "p0p100" using $wid2024, clear
}

append using `nataccounts1'
append using `nataccounts2'
append using `nataccounts3'

// sort country year variable 
// ren country iso
sort iso year widcode
// drop percentile age pop 
ren widcode variable

keep iso year variable value 

preserve
greshape wide value, i(iso year) j(variable) string
keep if year>1969 
greshape long value, i(iso year) j(variable) string
tab variable if value==. & year==2024
restore
greshape wide value, i(iso year) j(variable) string
renvars value*, predrop(6)
renvars *999i, postdrop(4)


keep if year>1969 
g corecountry=.
foreach country in $corecountries{
	replace corecountry=1 if iso=="`country'"
}
keep if corecountry==1
drop corecountry

tab iso
return list
	
save "$work_data/wid-database_privilege.dta", replace




use "$work_data/wid-database_privilege.dta", clear 
// sort iso year
encode iso, g(id)
xtset id year, yearly
sort iso year
gl all "nyixx comnx fdinx fdipx fdirx fdixa fdixd fkanx gdpro nninc nwgxa nwgxd pinpx pinrx ptdpx ptdrx ptdxa ptdxd ptepx pterx ptexa ptexd ptfnx ptfon ptfop ptfor ptfpx ptfrn ptfrp ptfrr ptfrx ptfxa ptfxd ptrrx ptrxa scinx taxnx tbnnx tgnnx tsnnx popul lcusx"
gl all_nopop "nyixx comnx fdinx fdipx fdirx fdixa fdixd fkanx gdpro nninc nwgxa nwgxd pinpx pinrx ptdpx ptdrx ptdxa ptdxd ptepx pterx ptexa ptexd ptfnx ptfon ptfop ptfor ptfpx ptfrn ptfrp ptfrr ptfrx ptfxa ptfxd ptrrx ptrxa scinx taxnx tbnnx tgnnx tsnnx  lcusx"
gl indexes "lcusx nyixx"
gl macro " comnx fdinx fdipx fdirx fdixa fdixd fkanx gdpro nninc nwgxa nwgxd pinpx pinrx ptdpx ptdrx ptdxa ptdxd ptepx pterx ptexa ptexd ptfnx ptfon ptfop ptfor ptfpx ptfrn ptfrp ptfrr ptfrx ptfxa ptfxd ptrrx ptrxa scinx taxnx tbnnx tgnnx tsnnx  "
foreach var in $all_nopop {
	replace `var'=. if year==2024
}
sort id year  
foreach v in $indexes {
//     by id (year): replace `v' = (L1.`v' + L2.`v')/2 if year==2024
by id (year): replace `v' = L1.`v' if year==2024
}
sort id year
foreach v in $macro {
//   by id (year): replace `v' = (L1.`v' + L2.`v' + L3.`v' + L4.`v' + L5.`v' + L6.`v' + L7.`v' + L8.`v' + L9.`v' + L10.`v')/10  if year==2024
by id (year): replace `v' = L1.`v' if year==2024
}
drop id
save "$work_data/wid-database_privilege.dta", replace

**# CREATE DATASET AT COUNTRY LEVEL 
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         CREATE DATASET AT COUNTRY LEVEL           *
* ---------------------------------------------------------------------------- * 
//CREATE DATASET AT COUNTRY LEVEL 

use "$work_data/wid-database_privilege.dta", clear 

foreach var in fdinx fdipx fdirx gdpro ptfnx ptfon ptfop ptfor ptfpx ptfrn ptfrp ptfrr ptfrx pinrx pinpx nninc nwgxa nwgxd fdixa fdixd ptfxa ptfxd {
	gen `var'_idx = `var'*nyixx
	replace `var' = `var'_idx/lcusx
}




gen nwnxa = nwgxa - nwgxd

// wealth stock is the one from the previous period
encode iso, gen(i)
xtset i year 
foreach var in nwgxa nwnxa nwgxd fdixa fdixd ptfxa ptfxd {
	gen orig`var' = `var'
	gen aux`var' = l.`var'
	replace `var' = aux`var'
}
drop i aux*


// return rates

gen r_a = pinrx/nwgxa
gen r_l = pinpx/nwgxd
gen rf_a = fdirx/fdixa
gen rf_l = fdipx/fdixd
gen rp_a = ptfrx/(ptfxa)
gen rp_l = ptfpx/ptfxd

// excess yields

gen priv = r_a - r_l

// NFKI
gen pinnx = (pinrx - pinpx)/gdpro 

// Excess yields as a share of GDP
gen exc =  priv*nwgxa if priv > 0
replace exc = priv*nwgxd if priv < 0
replace exc =  exc/gdpro

preserve 
gsort iso -year 
bys iso : carryforward exc if year ==1970, replace
keep iso year exc
save "$work_data/exc_country.dta", replace
restore 


//Express wealth variable as share of gdp 


foreach var in orignwnxa orignwgxd  orignwgxa  {
	gen `var'_gdp = `var'/gdpro
	
}
rename orig*_gdp *_gdp

save "$work_data/data_country.dta", replace

//CREATE DATASET AT QUINTILE LEVEL 
**# CREATE DATASET AT QUINTILE LEVEL 

use "$work_data/wid-database_privilege.dta", clear 

foreach var in fdinx fdipx fdirx gdpro ptfnx ptfon ptfop ptfor ptfpx ptfrn ptfrp ptfrr ptfrx pinrx pinpx nninc nwgxa nwgxd fdixa fdixd ptfxa ptfxd {
	gen `var'_idx = `var'*nyixx
	replace `var' = `var'_idx/lcusx
}


gen anninc = nninc/popul
gen agdp = gdpro/popul

gen aux = nyixx if iso == "US"
bys year : egen index_US = mode(aux)

gen nwnxa = nwgxa - nwgxd

// wealth stock is the one from the previous period
encode iso, gen(i)
xtset i year 
foreach var in nwgxa nwgxd nwnxa fdixa fdixd ptfxa ptfxd {
	gen orig`var' = `var'
	gen aux`var' = l.`var'
	replace `var' = aux`var'
}
drop i aux*


gen weight = popul  


//distribution ranking based on national income 

deciles, variable(anninc) by(year)

****instead of deciles use xtile
* Ensure data is sorted
sort year anninc
* Generate quintiles by year using population weights
// egen qanninc = xtile(anninc), by(year) nquantiles(5) weight(popul)
// egen danninc = xtile(anninc), by(year) nquantiles(10) weight(popul)

* qanninc is now in your dataset

preserve
keep if year==2022
keep iso year qanninc weight
collapse (sum) weight, by(qanninc)
gen cum_pop = sum(weight)
sort qanninc
sum cum_pop if qanninc==5
g share=.
forv x=1/5{
	replace share=weight/`r(mean)'
}

// qanninc	popul	cum_pop	share
// 1	1.621e+09	1.62e+09	.2020941
// 2	1.644e+09	3.27e+09	.2050334
// 3	1.570e+09	4.84e+09	.1957319
// 4	1.716e+09	6.55e+09	.2140152
// 5	1.469e+09	8.02e+09	.1831254



restore


preserve 
	keep iso year weight popul qanninc danninc 
	sa "$work_data\quintiles", replace
restore 


foreach var in pinrx pinpx nwnxa nwgxa nwgxd gdpro fdirx fdixa ptfrx  ptfxa  fdipx fdixd ptfpx ptfxd ///
            orignwgxa orignwgxd orignwnxa origfdixa origfdixd origptfxa origptfxd {

replace `var' = `var'*weight/popul

} 

collapse (sum) pinrx pinpx nwnxa nwgxa nwgxd gdpro fdirx fdixa ptfrx ptfxa fdipx fdixd ptfpx ptfxd weight ///
               orignwgxa orignwgxd orignwnxa origfdixa origfdixd origptfxa origptfxd, by(qanninc year)

foreach var in pinrx pinpx nwnxa nwgxa nwgxd gdpro fdirx fdixa ptfrx ptfxa  fdipx  fdixd ptfpx ptfxd weight ///
               orignwgxa orignwgxd orignwnxa origfdixa origfdixd origptfxa origptfxd {
	* world average
bys year : egen tot`var' = total(`var')
}

sort year qanninc

//Create wealth graphs variables 

//Global 

gen gfl_wld   = totorignwgxd/totgdpro
gen gfa_wld   = totorignwgxa/totgdpro
gen nwnxa_wld = gfa_wld - gfl_wld
	
	
//Quintiles gdp 

gen gfa_quint   = orignwgxa/gdpro
gen gfl_quint   = orignwgxd/gdpro
gen nwnxa_quint = orignwnxa/gdpro

//Global gdp as denominator 

gen gfaw   = orignwgxa/totgdpro
gen gflw   = orignwgxd/totgdpro
gen nwnxaw = orignwnxa/totgdpro

//Create returns variables

gen r_a_wld = totpinrx/totnwgxa
gen r_l_wld = totpinpx/totnwgxd

gen rf_a_wld  = totfdirx/totfdixa
gen rp_a_wld  = totptfrx/totptfxa

gen rf_l_wld = totfdipx/totfdixd

gen r_a_quint = pinrx/nwgxa
gen r_l_quint = pinpx/nwgxd
	
gen rf_a_quint = fdirx/fdixa
gen rp_a_quint = ptfrx/ptfxa

gen rf_l_quint = fdipx/fdixd
gen rp_l_quint = ptfpx/ptfxd

gen ratio_ptfxa_quint = ptfxa/nwgxa
gen ratio_fdixa_quint = fdixa/nwgxa
gen ratio_ptfxd_quint = ptfxd/nwgxd
gen ratio_fdixd_quint = fdixd/nwgxd

gen tot_pinnx_quint = pinrx - pinpx
	
gen priv_wld = r_a_wld - r_l_wld 

gen priv_quint= r_a_quint- r_l_quint


*excess yield as a share of GDP

gen exc_quint =  priv_quint*nwgxa if priv_quint > 0
replace exc_quint = priv_quint*nwgxd if priv_quint < 0
replace exc_quint=  exc_quint/gdpro

preserve 
keep qanninc year exc_quint 
ren exc_quint exc 
gsort qanninc -year 
bys qanninc : carryforward exc if year ==1970, replace
save "$work_data/exc_quint.dta", replace
restore 

gen excw_quint = exc_quint/totgdpro


*net capital income as a share of GDP

gen tot_pinnx_quintw = tot_pinnx_quint/totgdpro
replace tot_pinnx_quint = tot_pinnx_quint/gdpro

*percentage of net capital income due to excess yield 

gen absexc_quint = abs(exc_quint)
gen abstot_pinnx_quint= abs(tot_pinnx_quint)
gen excpinnx_quint= tot_pinnx_quint-exc_quint


 // differential wrt world average

gen difra_quint = r_a_quint - r_a_wld
gen difrl_quint = r_l_quint - r_l_wld
gen difexcess_quint = difra_quint*nwgxa - difrl_quint*nwgxd
replace difexcess_quint = difexcess_quint/gdpro
gen difexcess_quintw = difexcess_quint/totgdpro

save "$work_data/data_quintiles.dta", replace


//CREATE DATASET AT REGIONAL LEVEL 
**# CREATE DATASET AT REGIONAL LEVEL 

use "$work_data/wid-database_privilege.dta", clear 

foreach var in fdinx fdipx fdirx gdpro ptfnx ptfon ptfop ptfor ptfpx ptfrn ptfrp ptfrr ptfrx pinrx pinpx nninc nwgxa nwgxd fdixa fdixd ptfxa ptfxd {
	gen `var'_idx = `var'*nyixx
	replace `var' = `var'_idx/lcusx
}

gen nwnxa = nwgxa - nwgxd 

gen anninc = nninc/popul
gen agdp = gdpro/popul

gen aux = nyixx if iso == "US"
bys year : egen index_US = mode(aux)

// wealth stock is the one from the previous period
encode iso, gen(i)
xtset i year 
foreach var in nwgxa nwgxd nwnxa fdixa fdixd ptfxa ptfxd {
	gen orig`var' = `var'
	gen aux`var' = l.`var'
	replace `var' = aux`var'
}
drop i aux*

// merge m:1 iso using "$raw/import-country-codes-output.dta", nogen keepusing(region5) keep(1 3)
//
// replace region5 = "Latin America" 	if iso == "BQ"
// replace region5 = "Europe" 			if iso == "KS"
// replace region5 = "Europe" 			if iso == "GG"
// replace region5 = "Europe" 			if iso == "JE"
// ren region5 region 

	g region=""
foreach reg in MENA RUCA SSAF SSEA EURO LATA NAOC EASA{
	foreach c in $`reg'{
		replace region="`reg'" if iso=="`c'"
	}	
}

replace region = "MENA"   if region == "MENA"
replace region = "Russia and Central Asia"        if region == "RUCA"
replace region = "Sub-Saharan Africa"           if region == "SSAF"
replace region = "South and Southeast Asia"      if region == "SSEA"
replace region = "Europe"                       if region == "EURO"
replace region = "Latin America"                if region == "LATA"
replace region = "North America and Oceania"      if region == "NAOC"
replace region = "East Asia"                    if region == "EASA"

replace region = "China" if iso == "CN"


collapse (sum)  pinrx pinpx nwgxa nwgxd gdpro fdirx fdixa ptfrx ptfxa fdipx fdixd ptfpx ptfxd ///
                orignwgxa orignwgxd orignwnxa origfdixa origfdixd origptfxa origptfxd, by(region year)
 

// return rates
gen r_a = pinrx/nwgxa

gen rf_a = fdirx/fdixa
gen rp_a = ptfrx/ptfxa

gen r_l = pinpx/nwgxd
gen rf_l = fdipx/fdixd
gen rp_l = ptfpx/ptfxd

// excess yields
gen priv = r_a - r_l

*net capital income as a share of GDP
bys year : egen totgdpro = total(gdpro)

gen tot_pinnx_w = (pinrx - pinpx)/totgdpro
gen tot_pinnx = (pinrx - pinpx)/gdpro 

gen exc =  priv*nwgxa if priv > 0
replace exc = priv*nwgxd if priv < 0

preserve 
keep region year exc
gsort region -year 
bys region : carryforward exc if year ==1970, replace
save "$work_data/exc_region.dta", replace
restore 

replace exc =  exc/gdpro

gen excpinnx = tot_pinnx-exc

//Express wealth variable as share of gdp 

foreach var in orignwgxa orignwnxa orignwgxd  {
	gen `var'_gdp = `var'/gdpro
}

rename orig*_gdp *_gdp

encode region, gen(reg)

save "$work_data/data_regions.dta", replace




**# Figure 5.1. The U.S. exorbitant privilege has evolved into a structural privilege of the rich world
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure 1: US privilege has become a Rich world privilege, financed by the BRICS (Nievas and Sodano (2025))        *
* ---------------------------------------------------------------------------- * 

*********************************************************
//Additional G8 vs Brics  
*********************************************************

use "$work_data/data_country.dta", clear 
drop if year == 1970
// drop if year>2022

gen euro = .

replace euro = 1 if iso == "BE" // & year >= 1999
replace euro = 1 if iso == "DE" // & year >= 1999
replace euro = 1 if iso == "ES" // & year >= 1999
replace euro = 1 if iso == "FR" // & year >= 1999
replace euro = 1 if iso == "IE" // & year >= 1999
replace euro = 1 if iso == "IT" // & year >= 1999
replace euro = 1 if iso == "LU" // & year >= 1999
replace euro = 1 if iso == "NL" // & year >= 1999
replace euro = 1 if iso == "AT" // & year >= 1999
replace euro = 1 if iso == "PT" // & year >= 1999
replace euro = 1 if iso == "FI" // & year >= 1999
replace euro = 1 if iso == "GR" & year >= 2001
replace euro = 1 if iso == "SI" & year >= 2007
replace euro = 1 if iso == "CY" & year >= 2008
replace euro = 1 if iso == "MT" & year >= 2008
replace euro = 1 if iso == "SK" & year >= 2009
replace euro = 1 if iso == "EE" & year >= 2011
replace euro = 1 if iso == "LV" & year >= 2014
replace euro = 1 if iso == "LT" & year >= 2015

bys year : egen totpinrx = total(pinrx) if euro == 1
bys year : egen totnwgxa = total(nwgxa) if euro == 1
bys year : egen totpinpx = total(pinpx) if euro == 1
bys year : egen totnwgxd = total(nwgxd) if euro == 1
bys year : egen totgdpro = total(gdpro) if euro == 1 

gen euror_a = totpinrx/totnwgxa
gen euror_l = totpinpx/totnwgxd
// excess yields
gen europriv = euror_a - euror_l
// NFKI
gen totpinnx = (totpinrx - totpinpx)/totgdpro 
// Excess yields as a share of GDP
gen euroexc =  europriv*totnwgxa if europriv > 0
replace euroexc = europriv*totnwgxd if europriv < 0
replace euroexc =  euroexc/totgdpro
drop tot*

gen brics = . 
replace brics = 1 if iso == "BR"
replace brics = 1 if iso == "RU"
replace brics = 1 if iso == "IN"
replace brics = 1 if iso == "CN"
replace brics = 1 if iso == "ZA"

bys year : egen totpinrx = total(pinrx) if brics == 1
bys year : egen totnwgxa = total(nwgxa) if brics == 1
bys year : egen totpinpx = total(pinpx) if brics == 1
bys year : egen totnwgxd = total(nwgxd) if brics == 1
bys year : egen totgdpro = total(gdpro) if brics == 1 

gen bricsr_a = totpinrx/totnwgxa
gen bricsr_l = totpinpx/totnwgxd
// excess yields
gen bricspriv = bricsr_a - bricsr_l 
// NFKI
gen totpinnx = (totpinrx - totpinpx)/totgdpro 
// Excess yields as a share of GDP
gen bricsexc =  bricspriv*totnwgxa if bricspriv > 0
replace bricsexc = bricspriv*totnwgxd if bricspriv < 0
replace bricsexc =  bricsexc/totgdpro
drop tot*


keep if inlist(iso, "CA", "FR", "DE", "IT", "JP", "GB", "US") | ///
		inlist(iso, "BR", "RU", "IN", "CN", "ZA")



gsort iso year
mylabels -6(1)5, myscale(@/100) local(myla) suffix("%")
twoway (line exc year if iso == "JP", lwidth(medthick) lcolor("146 195 51")) 					 	///
	   (line exc year if iso == "US", lwidth(medthick) lcolor("0 103 167")) 						 	///
	   (line exc year if iso == "RU", lwidth(medthick) lcolor("255 99 51")) 						///
	   (line exc year if iso == "CN", lwidth(medthick) lcolor("246 199 0")) 						///
	   (line euroexc year if iso == "FR", lc("212 17 89") lpattern(dash) lw(thick))								///
	   (line bricsexc year if iso == "BR", lc("15 39 75") lpattern(dash) lw(thick))							///
, title("") subtitle("") ytitle("Excess yield as % of country GDP") xtitle("") legend(order( 5 "Eurozone" 6 "BRICS" 2 "United States" 4 "China" 1 "Japan" 3 "Russia") rows(4) pos(8) ring(0) size(large) region(col(none))) ylabel(`myla', grid glcolor(gs15)) xlabel(1970(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3)) yline(0, lcolor(black) lpattern(dash))  

* Keep only the countries used in the graph
keep if inlist(iso, "JP", "US", "RU", "CN", "FR", "BR")
keep year iso exc euroexc bricsexc

* Create a single "value" variable combining the relevant source
gen exc_graph = .
replace exc_graph = exc if inlist(iso, "JP", "US", "RU", "CN")
replace exc_graph = euroexc if iso == "FR"
replace exc_graph = bricsexc if iso == "BR"

* Keep only what's needed
keep year iso exc_graph

* Reshape to wide format: one column per country
reshape wide exc_graph, i(year) j(iso) string

* Rename columns for clarity
ren exc_graphJP JP
ren exc_graphUS US
ren exc_graphRU RU
ren exc_graphCN CN
ren exc_graphFR Eurozone
ren exc_graphBR BRICS

insobs 1
replace year = 1970 if year==.
sort year

label var BRICS "BRICS"
label var CN "China"
label var Eurozone "Eurozone" 
label var JP "Japan"
label var RU "Russia"
label var US "United States"

drop year 
egen year =seq()
replace year=year+1969
order year
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-F5.1") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-F5.1")
putexcel (A2:A60), nformat("0")
putexcel (B2:H60), nformat(percent)




**# Figure 5.7.a. Poorer countries face lower asset returns and higher liability costs
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure 7: Returns on foreign assets per income group     (Nievas and Sodano (2025))     *
* ---------------------------------------------------------------------------- * 

use "$work_data\data_quintiles.dta", clear
drop if year == 1970
// drop if year > 2022

// Returns on foreign assets per income group
mylabels 0(1)11, myscale(@/100) local(myla) suffix("%")
twoway (line r_a_quint year if qanninc == 1, lwidth(medthick) lcolor("64 105 166"))			 	 ///
	   (line r_a_quint year if qanninc == 2, lwidth(medthick) lcolor("246 199 0"))			 	 ///	   
	   (line r_a_quint year if qanninc == 3, lwidth(medthick) lcolor("146 195 51"))			 	 ///
	   (line r_a_quint year if qanninc == 4, lwidth(medthick) lcolor("orange"))			 		 ///
	   (line r_a_quint year if qanninc == 5, lwidth(medthick) lcolor("200 0 0"))			  	 ///
	   (line r_a_wld year, lwidth(medthick) lcolor("black") lpattern(dash))  ///
	   , title("") subtitle("") ytitle("") xtitle("") legend(order(6 "World" 1 "Bottom 20%" 2 "20%-40%" 3 "40%-60%" 4 "60%-80%" 5 "Top 20%") rows(2) pos(bottom) size(large)) note("") ylabel(`myla', grid glcolor(gs15)) xlabel(1970(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3))
// Graph shows average rate of returns on foreign assets." "Simple averages by group. Countries grouped according to national income per capita quintiles, weighted by population." "National income does not include FDI income paid correction due to shifted profits.


* Keep only the variables needed
keep year qanninc r_a_quint r_a_wld

* Drop rows where r_a_quint is missing and qanninc is not in 1–5
keep if inlist(qanninc, 1, 2, 3, 4, 5) | missing(qanninc)

* Isolate World average for merging later
preserve
    keep if missing(qanninc)
    keep year r_a_wld
    tempfile world
    save `world'
restore

* Keep only quintile-specific data
keep if inlist(qanninc, 1, 2, 3, 4, 5)

* Reshape to wide format: one column per quintile
reshape wide r_a_quint, i(year) j(qanninc)

* Rename columns for clarity
ren r_a_quint1 Bottom_20
ren r_a_quint2 Q20_40
ren r_a_quint3 Q40_60
ren r_a_quint4 Q60_80
ren r_a_quint5 Top_20

* Merge world average
merge 1:1 year using `world', nogen

* Optional: Rename world column
ren r_a_wld World

insobs 1
replace year = 1970 if year==.
sort year

label var Bottom_20  "Bottom 20%"
label var Q20_40  "20%-40%"
label var Q40_60  "40%-60%" 
label var Q60_80  "60%-80%"
label var Top_20 "Top 20%"

drop year 
egen year =seq()
replace year=year+1969
order year
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-F5.7.a.") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-F5.7.a.")
putexcel (B2:H60), nformat(percent)


**# Figure 5.7.b. Poorer countries face lower asset returns and higher liability costs
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure 8: Returns on foreign liabilities per income group   (Nievas and Sodano (2025))      *
* ---------------------------------------------------------------------------- * 
use "$work_data\data_quintiles.dta", clear
drop if year == 1970
*liabilities
// Returns on foreign liabilities per income group
mylabels 0(1)11, myscale(@/100) local(myla) suffix("%")
twoway (line r_l_quint year if qanninc == 1, lwidth(medthick) lcolor("64 105 166"))			 	 ///
	   (line r_l_quint year if qanninc == 2, lwidth(medthick) lcolor("246 199 0"))			 	 	 ///	   
	   (line r_l_quint year if qanninc == 3, lwidth(medthick) lcolor("146 195 51"))			 	 ///
	   (line r_l_quint year if qanninc == 4, lwidth(medthick) lcolor("orange"))			 		 ///
	   (line r_l_quint year if qanninc == 5, lwidth(medthick) lcolor("200 0 0"))			  		 ///
	   (line r_l_wld year, lwidth(medthick) lcolor("black") lpattern(dash)) 	 ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(6 "World" 1 "Bottom 20%" 2 "20%-40%" 3 "40%-60%" 4 "60%-80%" 5 "Top 20%") rows(2) pos(bottom) size(large)) note("") ylabel(`myla', grid glcolor(gs15)) xlabel(1970(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3))
// Graph shows average rate of returns on foreign liabilities." "Simple averages by group. Countries grouped according to national income per capita quintiles, weighted by population." "National income does not include FDI income paid correction due to shifted profits.


* Keep only the variables needed
keep year qanninc r_l_quint r_l_wld

* Drop rows where r_l_quint is missing and qanninc is not in 1–5
keep if inlist(qanninc, 1, 2, 3, 4, 5) | missing(qanninc)

* Isolate World average for merging later
preserve
    keep if missing(qanninc)
    keep year r_l_wld
    tempfile world
    save `world'
restore

* Keep only quintile-specific data
keep if inlist(qanninc, 1, 2, 3, 4, 5)

* Reshape to wide format: one column per quintile
reshape wide r_l_quint, i(year) j(qanninc)

* Rename columns for clarity
ren r_l_quint1 Bottom_20
ren r_l_quint2 Q20_40
ren r_l_quint3 Q40_60
ren r_l_quint4 Q60_80
ren r_l_quint5 Top_20

* Merge world average
merge 1:1 year using `world', nogen

* Optional: Rename world column
ren r_l_wld World

insobs 1
replace year = 1970 if year==.
sort year

label var Bottom_20  "Bottom 20%"
label var Q20_40  "20%-40%"
label var Q40_60  "40%-60%" 
label var Q60_80  "60%-80%"
label var Top_20 "Top 20%"

drop year 
egen year =seq()
replace year=year+1969
order year
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-F5.7.b.") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-F5.7.b.")
putexcel (B2:H60), nformat(percent)




**# Figure 5.2.a. Rich countries receive 1% of their GDP from the rest of the world due to financial privilege
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure 2: Excess yield income as a share of GDP    (Nievas and Sodano (2025))     *
* ---------------------------------------------------------------------------- * 
use "$work_data\data_quintiles.dta", clear
drop if year == 1970
*excess yield as a share of GDP

 // Excess yield as a share of GDP
mylabels -4(1)1, myscale(@/100) local(myla) suffix("%")
twoway (line exc_quint year if qanninc == 1, lwidth(medthick) lcolor("64 105 166"))			 		 ///
	   (line exc_quint year if qanninc == 2, lwidth(medthick) lcolor("246 199 0"))			 			 ///	   
	   (line exc_quint year if qanninc == 3, lwidth(medthick) lcolor("146 195 51"))			 		 ///
	   (line exc_quint year if qanninc == 4, lwidth(medthick) lcolor("orange"))			 		   	 ///
	   (line exc_quint year if qanninc == 5, lwidth(medthick) lcolor("200 0 0"))			  		 ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(1 "Bottom 20%" 2 "20%-40%" 3 "40%-60%" 4 "60%-80%" 5 "Top 20%") rows(2) pos(7) ring(0) size(large)) note("") ylabel(`myla', grid glcolor(gs15)) xlabel(1970(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3)) yline(0, lcolor(black) lpattern(dash)) 
// Graph shows GFA multiplied by excess yield if positive and GFL multiplied by excess yield if negative" "Countries grouped according to national income per capita quintiles, weighted by population." "National income does not include FDI income paid correction due to shifted profits.

// br  exc_quint gdpro qanninc year  if year==2023
g revenue= exc_quint*gdpro 
sum revenue if year==2023 & qanninc==5
loc revenue=r(mean) 

*perspective
// preserve
			import excel "$raw/imf-dm-export-20250721.xls", clear firstrow

			rename CentralGovernmentDebtPercent country

			keep country BW
			ren BW X2023
			drop if country==""
			replace X2023="" if X2023=="no data"
			destring X2023,replace
			ren country isoname 

			tempfile imf
			save `imf',replace


			use if 	 inrange(year,2023,2023) & p=="p0p100" & ///
					inlist(widcode, "xlcusp999i") ///
					using $wid2024, clear
			g corecountry=.
			foreach country in $corecountries{
				replace corecountry=1 if iso=="`country'"
			}
			keep if corecountry==1
			drop corecountry p currency	
			keep year iso value	widcode	
			greshape wide value, i(year iso) j(widcode) string
			renvars value*, pred(5)
			drop year
			tempfile ppp2023
			save `ppp2023',replace
					

			use if inlist(widcode,"mrevgo999i","mgdpro999i","npopul999i","mntrgr999i ") & inrange(year,1980, $year_output) /// 
				using $wid2024, clear
			tab year


			g corecountry=.
			foreach country in $corecountries{
				replace corecountry=1 if iso=="`country'"
			}
			keep if corecountry==1
			drop corecountry p currency

			* Reshape wide so we can work with both variables
			greshape wide value, i(year iso) j(widcode) string
			renvars value*, predrop(5)


			merge m:1 iso using `ppp2023'
			drop _merge


			replace mgdpro999i=mgdpro999i/xlcusp999i
			keep if year==2023
			keep year iso mgdpro999i npopul999i
			g gdp_pc=mgdpro999i/npopul999i
			replace mgdpro999i=mgdpro999i/1000000000

			preserve	
			import excel using "$raw/country-codes-new.xlsx", firstrow clear
			renvars code shortname / iso isoname
			keep iso isoname 
			replace isoname="Turkiye" if iso=="TR"
			replace isoname="Czechia" if iso=="CZ"
			replace isoname="Russia" if iso=="RU"
			replace isoname="Eswatini" if iso=="SZ"
			replace isoname="Syria" if iso=="SY"
			replace isoname="South Korea" if iso=="KR"
			tempfile isoname 
			save `isoname', replace	
			restore
			merge 1:1 iso using `isoname'
			drop _merge
			drop if year==.

			merge 1:1 isoname using `imf'
			drop _merge
			ren X2023 Central_Government_Debt
			g Debt=(Central_Government_Debt/100)*mgdpro999i
			gsort -Debt 

				g region=""
			foreach reg in MENA RUCA SSAF SSEA EURO LATA NAOC EASA{
				foreach c in $`reg'{
					replace region="`reg'" if iso=="`c'"
				}	
			}

			replace region = "MENA"   if region == "MENA"
			replace region = "Russia and Central Asia"        if region == "RUCA"
			replace region = "Sub-Saharan Africa"           if region == "SSAF"
			replace region = "South and Southeast Asia"      if region == "SSEA"
			replace region = "Europe"                       if region == "EURO"
			replace region = "Latin America"                if region == "LATA"
			replace region = "North America and Oceania"      if region == "NAOC"
			replace region = "East Asia"                    if region == "EASA"

			gsort gdp_pc -Debt 
			drop if Debt==.
			keep if npopul999i>10000000

			gen cum_debt = sum(Debt)
			gen cum_pop = sum(npopul999i)
			replace cum_pop=cum_pop/1000000

			tab isoname if cum_debt<360
			g revenue=`revenue'/1000000000
			tab isoname if cum_debt<750

// restore

use "$work_data\data_quintiles.dta", clear
drop if year == 1970		
* Keep only the variables needed
keep year qanninc exc_quint 

* Drop rows where exc_quint is missing and qanninc is not in 1–5
keep if inlist(qanninc, 1, 2, 3, 4, 5) | missing(qanninc)



* Keep only quintile-specific data
keep if inlist(qanninc, 1, 2, 3, 4, 5)

* Reshape to wide format: one column per quintile
reshape wide exc_quint, i(year) j(qanninc)

* Rename columns for clarity
ren exc_quint1 Bottom_20
ren exc_quint2 Q20_40
ren exc_quint3 Q40_60
ren exc_quint4 Q60_80
ren exc_quint5 Top_20





insobs 1
replace year = 1970 if year==.
sort year

label var Bottom_20  "Bottom 20%"
label var Q20_40  "20%-40%"
label var Q40_60  "40%-60%" 
label var Q60_80  "60%-80%"
label var Top_20 "Top 20%"

drop year 
egen year =seq()
replace year=year+1969
order year
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-F5.2.a.") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-F5.2.a.")
putexcel (B2:H60), nformat(percent)



**# Figure 5.4.a There is a persistent net income transfer from poor to rich
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure 6: Net foreign capital income as a share of GDP  (Nievas and Sodano (2025))      *
* ---------------------------------------------------------------------------- * 

use "$work_data\data_quintiles.dta", clear
drop if year == 1970
*net capital income as a share of GDP

// Net foreign capital income as a share of GDP
mylabels -4(1)2, myscale(@/100) local(myla) suffix("%")
twoway (line tot_pinnx_quint year if qanninc == 1, lwidth(medthick) lcolor("64 105 166"))			 	 ///
	   (line tot_pinnx_quint year if qanninc == 2, lwidth(medthick) lcolor("246 199 0"))			 		 ///	   
	   (line tot_pinnx_quint year if qanninc == 3, lwidth(medthick) lcolor("146 195 51"))			 	 ///
	   (line tot_pinnx_quint year if qanninc == 4, lwidth(medthick) lcolor("orange"))			 		 ///
	   (line tot_pinnx_quint year if qanninc == 5, lwidth(medthick) lcolor("200 0 0"))			  	 ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(1 "Bottom 20%" 2 "20%-40%" 3 "40%-60%" 4 "60%-80%" 5 "Top 20%") rows(2) pos(12) ring(0) size(large)) note("") ylabel(`myla', grid glcolor(gs15)) xlabel(1970(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3)) yline(0, lcolor(black) lpattern(dash)) 
// Graph shows aggregate net foreign capital income, as a share of income group GDP." "Countries grouped according to national income per capita quintiles, weighted by population." "National income does not include FDI income paid correction due to shifted profits.

* Keep only the variables needed
keep year qanninc tot_pinnx_quint

* Drop rows where tot_pinxx_quint is missing and qanninc is not in 1–5
keep if inlist(qanninc, 1, 2, 3, 4, 5) | missing(qanninc)



* Keep only quintile-specific data
keep if inlist(qanninc, 1, 2, 3, 4, 5)

* Reshape to wide format: one column per quintile
reshape wide tot_pinnx_quint, i(year) j(qanninc)

* Rename columns for clarity
ren tot_pinnx_quint1 Bottom_20
ren tot_pinnx_quint2 Q20_40
ren tot_pinnx_quint3 Q40_60
ren tot_pinnx_quint4 Q60_80
ren tot_pinnx_quint5 Top_20



insobs 1
replace year = 1970 if year==.
sort year

label var Bottom_20  "Bottom 20%"
label var Q20_40  "20%-40%"
label var Q40_60  "40%-60%" 
label var Q60_80  "60%-80%"
label var Top_20 "Top 20%"

drop year 
egen year =seq()
replace year=year+1969
order year
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-F5.4.a") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-F5.4.a")
putexcel (B2:H60), nformat(percent)





**# Figure 5.4.b. There is a persistent net income transfer from poor to rich
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure A28: Net foreign capital income as a share of GDP  (Nievas and Sodano (2025))    *
* ---------------------------------------------------------------------------- * 
use "$work_data\data_regions.dta", clear 

drop if year == 1970
// Net foreign capital income as a share of GDP
mylabels -10(1)4, myscale(@/100) local(myla) suffix("%")
twoway (line tot_pinnx year if reg == 1, lwidth(medthick) lcolor("orange")) 					 ///
	   (line tot_pinnx year if reg == 2, lwidth(medthick) lcolor("246 199 0"))					 ///
	   (line tot_pinnx year if reg == 3, lwidth(medthick) lcolor("200 0 0"))			 		 ///	   
	   (line tot_pinnx year if reg == 4, lwidth(medthick) lcolor("dknavy"))			 			 ///
	   (line tot_pinnx year if reg == 5, lwidth(medthick) lcolor("green")) 						 ///
	   (line tot_pinnx year if reg == 6, lwidth(medthick) lcolor("146 195 51"))			  		 ///
	   (line tot_pinnx year if reg == 7, lwidth(medthick) lcolor("64 105 166")) 						 ///
	   (line tot_pinnx year if reg == 8, lwidth(medthick) lcolor("brown")) 				 		 ///
	   (line tot_pinnx year if reg == 9, lwidth(medthick) lcolor("teal")) 				 		 ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(1 "China" 2 "East Asia (excluding China)" 3 "Europe" 4 "Latin America" 5 "Middle East & North Africa" 6 "North America & Oceania" 7 "Russia & Central Asia" 8 "South & South-East Asia" 9 "Subsaharan Africa") rows(3) pos(bottom)) note("") ylabel(`myla', grid glcolor(gs15)) xlabel(1970(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3)) yline(0, lcolor(black) lpattern(dash)) 

* Keep only data used in the graph
keep if inlist(reg, 1,2,3,4,5,6,7,8,9)
keep year reg tot_pinnx

* Reshape to wide format: one column per region
reshape wide tot_pinnx, i(year) j(reg)

* Rename columns for clarity (based on your legend)
ren tot_pinnx1  China
ren tot_pinnx2  East_Asia_ex_China
ren tot_pinnx3  Europe
ren tot_pinnx4  Latin_America
ren tot_pinnx5  MENA
ren tot_pinnx6  North_America_Oceania
ren tot_pinnx7  Russia_Central_Asia
ren tot_pinnx8  South_SE_Asia
ren tot_pinnx9  SubSaharan_Africa

* Optional: insert a dummy row for 1970 (not plotted, but consistent with previous)
insobs 1
replace year = 1970 if missing(year)

sort year

* Label columns nicely for tot_pinnxel
label var China                 "China"
label var East_Asia_ex_China   "East Asia (excluding China)"
label var Europe               "Europe"
label var Latin_America        "Latin America"
label var MENA                 "Middle East & North Africa"
label var North_America_Oceania "North America & Oceania"
label var Russia_Central_Asia  "Russia & Central Asia"
label var South_SE_Asia        "South & South-East Asia"
label var SubSaharan_Africa    "Sub-Saharan Africa"

* Export to Excel
drop year 
egen year =seq()
replace year=year+1969
order year
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-F5.4.b.") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-F5.4.b.")
putexcel (B2:K60), nformat(percent)


**# Figure 5.2.b. Rich countries receive 1% of their GDP from the rest of the world due to financial privilege
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure A27: Excess yield as a share of GDP  (Nievas and Sodano (2025))   *
* ---------------------------------------------------------------------------- * 
use "$work_data\data_regions.dta", clear 

drop if year == 1970
// Excess yields as a share of GDP
mylabels -10(1)3, myscale(@/100) local(myla) suffix("%")
twoway (line exc year if reg == 1, lwidth(medthick) lcolor("orange")) 					 ///
	   (line exc year if reg == 2, lwidth(medthick) lcolor("246 199 0"))					 ///
	   (line exc year if reg == 3, lwidth(medthick) lcolor("200 0 0"))			 		 ///	   
	   (line exc year if reg == 4, lwidth(medthick) lcolor("dknavy"))			 			 ///
	   (line exc year if reg == 5, lwidth(medthick) lcolor("green")) 						 ///
	   (line exc year if reg == 6, lwidth(medthick) lcolor("146 195 51"))			  		 ///
	   (line exc year if reg == 7, lwidth(medthick) lcolor("64 105 166")) 						 ///
	   (line exc year if reg == 8, lwidth(medthick) lcolor("brown")) 				 		 ///
	   (line exc year if reg == 9, lwidth(medthick) lcolor("teal")) 				 		 ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(1 "China" 2 "East Asia (excluding China)" 3 "Europe" 4 "Latin America" 5 "Middle East & North Africa" 6 "North America & Oceania" 7 "Russia & Central Asia" 8 "South & South-East Asia" 9 "Subsaharan Africa") rows(3) pos(bottom)) note("") ylabel(`myla', grid glcolor(gs15)) xlabel(1970(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3)) yline(0, lcolor(black) lpattern(dash)) 
* Keep only data used in the graph
keep if inlist(reg, 1,2,3,4,5,6,7,8,9)
keep year reg exc

* Reshape to wide format: one column per region
reshape wide exc, i(year) j(reg)

* Rename columns for clarity (based on your legend)
ren exc1  China
ren exc2  East_Asia_ex_China
ren exc3  Europe
ren exc4  Latin_America
ren exc5  MENA
ren exc6  North_America_Oceania
ren exc7  Russia_Central_Asia
ren exc8  South_SE_Asia
ren exc9  SubSaharan_Africa

* Optional: insert a dummy row for 1970 (not plotted, but consistent with previous)
insobs 1
replace year = 1970 if missing(year)

sort year

* Label columns nicely for Excel
label var China                 "China"
label var East_Asia_ex_China   "East Asia (excluding China)"
label var Europe               "Europe"
label var Latin_America        "Latin America"
label var MENA                 "Middle East & North Africa"
label var North_America_Oceania "North America & Oceania"
label var Russia_Central_Asia  "Russia & Central Asia"
label var South_SE_Asia        "South & South-East Asia"
label var SubSaharan_Africa    "Sub-Saharan Africa"

* Export to Excel
drop year 
egen year =seq()
replace year=year+1969
order year
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-F5.2.b.") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-F5.2.b.")
putexcel (B2:K60), nformat(percent)







**# Figure 5.3.a. Privilege persists for the U.S. (and its region) despite negative net foreign asset positions
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure 3: Net foreign assets as a share of world GDP   (Nievas and Sodano (2025))  *
* ---------------------------------------------------------------------------- * 
use "$work_data/data_quintiles.dta", clear

* --- Adjust nwnxaw for 2024  by income quintile ---
sort qanninc year

keep year qanninc nwnxaw nwnxa_wld
greshape wide nwnxa_wld nwnxaw, i(year) j(qanninc)


foreach q in 1 2 3 4 5 {
	replace nwnxaw`q' = (nwnxaw`q'[_n-2] + nwnxaw`q'[_n-1])/2 if year == 2024
	
	replace nwnxa_wld`q' = (nwnxa_wld`q'[_n-2] + nwnxa_wld`q'[_n-1])/2 if year == 2024
}

greshape long nwnxa_wld nwnxaw, i(year) j(qanninc)

// Net foreign assets (corrected for offshore wealth) per income group as share of global GDP
mylabels -4(2)6, myscale(@/100) local(myla) suffix("%")
twoway (line nwnxaw year if qanninc == 1, lwidth(medthick) lcolor("64 105 166"))			 ///
	   (line nwnxaw year if qanninc == 2, lwidth(medthick) lcolor("246 199 0"))			 	 ///	   
	   (line nwnxaw year if qanninc == 3, lwidth(medthick) lcolor("146 195 51"))			 ///
	   (line nwnxaw year if qanninc == 4, lwidth(medthick) lcolor("orange"))			 	 ///
	   (line nwnxaw year if qanninc == 5, lwidth(medthick) lcolor("200 0 0"))			  	 ///
	   (line nwnxa_wld year, lwidth(medthick) lcolor("black") lpattern(dash))  ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(6 "World" 1 "Bottom 20%" 2 "20%-40%" 3 "40%-60%" 4 "60%-80%" 5 "Top 20%") rows(2)) note("") ylabel(`myla', grid glcolor(gs15)) xlabel(1970(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3))
// Graph shows average net foreign assets." "Simple averages by group. Countries grouped according to national income per capita quintiles, weighted by population." "National income does not include FDI income paid correction due to shifted profits.

greshape wide nwnxa_wld nwnxaw, i(year) j(qanninc)
keep year nwnxa_wld1 nwnxaw1 nwnxaw2 nwnxaw3 nwnxaw4 nwnxaw5
ren nwnxa_wld1 nwnxa_wld

label var nwnxa_wld    "World" 
label var nwnxaw1 "Bottom 20%"
label var nwnxaw2 "20%-40%"
label var nwnxaw3  "40%-60%" 
label var nwnxaw4  "60%-80%"
label var nwnxaw5 "Top 20%"



* Export to Excel
drop year 
egen year =seq()
replace year=year+1969
order year
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-5.3.a.") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-5.3.a.")
putexcel (A2:A60), nformat("0")
putexcel (B2:K60), nformat(percent)


**# Figure 5.3.b. Privilege persists for the U.S. (and its region) despite negative net foreign asset positions
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Nievas and Piketty (2025)
Figure 15 in Nievas and Piketty (2025)

Paper
https://wid.world/document/unequal-exchange-and-north-south-relations-evidence-from-global-trade-flows-and-the-world-balance-of-payments-1800-2025-world-inequality-lab-working-paper-2025-11/

Replication package
https://wid.world/document/data-file-to-unequal-exchange-and-north-south-relations-evidence-from-global-trade-flows-and-the-world-balance-of-payments-1800-2025-world-inequality-lab-working-paper-2025-11/
*/


**# Figure 5.6. Poor countries finance the privilege through asymmetric portfolios
* ---------------------------------------------------------------------------- *
* 				   	                                                       *
*         Figure 10: Rich countries hold less central bank reserves and less FDI liabilities   (Nievas and Sodano (2025))  *
* ---------------------------------------------------------------------------- * 
use "$work_data\wid-database_privilege.dta", clear

foreach var in fdinx fdipx fdirx gdpro ptfnx ptfon ptfop ptfor ptfpx ptfrn ptfrp ptfrr ptfrx pinrx pinpx nninc nwgxa nwgxd fdixa fdixd ptfxa ptfxd ptdrx pterx ptrrx ptdxa ptexa ptrxa ///
ptdpx ptepx ptdxd ptexd {
	gen `var'_idx = `var'*nyixx
	replace `var' = `var'_idx/lcusx
}

gen nwnxa = nwgxa - nwgxd 


// wealth stock is the one from the previous period
encode iso, gen(i)
xtset i year 
foreach var in nwgxa nwgxd fdixa fdixd ptfxa ptfxd ptexa ptexd ptdxa ptdxd ptrxa {
	gen orig`var' = `var'
	gen aux`var' = l.`var'
	replace `var' = aux`var'
}
drop i aux*
drop if year == 1970

merge 1:m iso year using "$work_data\quintiles", nogen

	
foreach var in pinrx pinpx  nwgxa nwnxa nwgxd gdpro fdirx fdixa ptfrx  ptfrp ptfrr ptfxa fdipx fdixd ptfpx ptfxd ptdrx pterx ptrrx ptdxa ptexa ptrxa ptdpx ptepx ptdxd ptexd {

replace `var' = `var'*weight/popul

} 

collapse (sum) pinrx pinpx  nwgxa nwnxa nwgxd gdpro fdirx fdixa ptfrx  ptfrp ptfrr ptfxa fdipx fdixd ptfpx ptfxd ptdrx pterx ptrrx ptdxa ptexa ptrxa ptdpx ptepx ptdxd ptexd, by(qanninc year)

preserve 
	collapse (sum) pinrx pinpx nwgxa nwnxa nwgxd gdpro fdirx fdixa ptfrx  ptfrp ptfrr ptfxa fdipx fdixd ptfpx ptfxd  ptdrx pterx ptrrx ptdxa ptexa ptrxa ptdpx ptepx ptdxd ptexd, by(year)
	gen qanninc = 99
	tempfile wld
	sa `wld'
restore

append using `wld'	

gen r_a = pinrx/nwgxa
gen r_l = pinpx/nwgxd 

	gen rf_a = fdirx/fdixa
	gen rp_a = ptfrx/ptfxa
		gen rpeq_a = (pterx + ptfrr)/ptexa  
		gen rpdeb_a = ptdrx/ptdxa
		gen rpres_a = ptrrx/ptrxa 
		
	gen rf_l = fdipx/fdixd
	gen rp_l = ptfpx/ptfxd
		gen rpeq_l = (ptepx + ptfrp)/ptexd 
		gen rpdeb_l = ptdpx/ptdxd 

	*rescaling change later
	gen ratiocheck = (ptexa + ptdxa + ptrxa)/ptfxa
	foreach var in ptexa ptdxa ptrxa {
		replace `var' = `var'/ratiocheck
	} 
	drop ratiocheck 
	
	gen ratio_ptfxa = ptfxa/nwgxa
		gen ratio_ptfxa_eq = ptexa/nwgxa
		gen ratio_ptfxa_deb = ptdxa/nwgxa
		gen ratio_ptfxa_res = ptrxa/nwgxa
	gen ratio_fdixa = fdixa/nwgxa
	
	
	gen ratio_ptfxd = ptfxd/nwgxd
		gen ratio_ptfxd_eq = ptexd/nwgxd
		gen ratio_ptfxd_deb = ptdxd/nwgxd
	gen ratio_fdixd = fdixd/nwgxd

	gen priv = r_a - r_l

// 	drop if year>2022 
	
//	return effect
	preserve
	collapse (mean) r_a* r_l* rf* rp* ratio* priv* nwgxa* nwgxd* fdixa fdixd ptfxa ptfxd ptdxa ptexa ptrxa ptdxd ptexd gdpro, by(qanninc) 
	tempfile decomposition 
	gen period = "all"
	sa `decomposition', replace
	restore
	preserve 
	keep if year <= 1999
	collapse (mean) r_a* r_l* rf* rp* ratio* priv* nwgxa* nwgxd* fdixa fdixd ptfxa ptfxd ptdxa ptexa ptrxa ptdxd ptexd gdpro, by(qanninc) 
	gen period = "1970-1999"
	append using `decomposition'
	sa `decomposition', replace
	restore

	keep if inrange(year,2000,2024) 
	collapse (mean) r_a* r_l* rf* rp* ratio* priv* nwgxa* nwgxd* fdixa fdixd ptfxa ptfxd ptdxa ptexa ptrxa ptdxd ptexd gdpro, by(qanninc) 
	gen period = "2000-2024"
	append using `decomposition'
	sa `decomposition', replace

*drop ptfxa 


reshape wide r_a r_l rf_a rf_l rp_a rp_l rpeq_a rpeq_l rpdeb_a rpdeb_l rpres_a ratio_ptfxa ratio_ptfxa_eq ratio_ptfxa_deb ratio_ptfxa_res ratio_fdixa ratio_ptfxd ratio_ptfxd_eq ratio_ptfxd_deb ratio_fdixd nwgxa nwgxd fdixa fdixd ptfxa ptexa ptdxa ptrxa ptfxd ptexd ptdxd gdpro priv, i(period) j(qanninc) 

forvalues i = 1/5 { 
	foreach x in r_a r_l rf_a rf_l rp_a rp_l rpeq_a rpeq_l rpdeb_a rpdeb_l rpres_a {
		gen diff_`x'`i' = `x'`i' - `x'99
	}
	
	foreach x in ratio_ptfxa ratio_ptfxa_eq ratio_ptfxa_deb ratio_ptfxa_res ratio_fdixa ratio_ptfxd ratio_ptfxd_eq ratio_ptfxd_deb ratio_fdixd {
		gen diff_`x'`i' = `x'`i' - `x'99
	}
	
	
// return 
	gen privwa`i' = (diff_r_a`i'*nwgxa`i')/gdpro`i'
	gen privwl`i' = -(diff_r_l`i'*nwgxd`i')/gdpro`i' // it is minus (-) because if I'm paying less in my liabilities than the world average then it contributes to my priv
	gen privw`i' = privwa`i' + privwl`i' 
	
	gen privwptfa`i' = (diff_rp_a`i'*ptfxa`i')/gdpro`i' 
	gen privwptfl`i' = -(diff_rp_l`i'*ptfxd`i')/gdpro`i'
	gen privwptf`i' = privwptfa`i' + privwptfl`i' 

		gen privwptfaeq`i' = (diff_rpeq_a`i'*ptexa`i')/gdpro`i' 
		gen privwptfleq`i' = -(diff_rpeq_l`i'*ptexd`i')/gdpro`i'
		gen privwptfeq`i' = privwptfaeq`i' + privwptfleq`i' 

		gen privwptfadeb`i' = (diff_rpdeb_a`i'*ptdxa`i') /gdpro`i' 
		gen privwptfldeb`i' = -(diff_rpdeb_l`i'*ptdxd`i')/gdpro`i'
		gen privwptfdeb`i' = privwptfadeb`i' + privwptfldeb`i' 
	
		gen privwptfares`i' = (diff_rpres_a`i'*ptrxa`i')/gdpro`i' 
		
	gen privwfdia`i' = (diff_rf_a`i'*fdixa`i')/gdpro`i' 
	gen privwfdil`i' = - (diff_rf_l`i'*fdixd`i')/gdpro`i' 
	gen privwfdi`i' = privwfdia`i' + privwfdil`i'

// composition 
*	gen compwa`i' = (r_a99*diff`i')/gdpro`i'
*	gen compwlc`i' = -(r_lc99*nwgxd`i')/gdpro`i'
*	gen compw`i' = compwa`i' + compwlc`i'
	
	gen compwptfa`i' = (rp_a`i'*diff_ratio_ptfxa`i')*ptfxa`i'/gdpro`i' 
	gen compwptfl`i' = -(rp_l`i'*diff_ratio_ptfxd`i')*ptfxd`i'/gdpro`i'
	gen compwptf`i' = compwptfa`i' + compwptfl`i' 
	
		gen compwptfaeq`i' = (rpeq_a`i'*diff_ratio_ptfxa_eq`i')*ptexa`i'/gdpro`i' 
		gen compwptfleq`i' = -(rpeq_l`i'*diff_ratio_ptfxd_eq`i')*ptexd`i'/gdpro`i'
		gen compwptfeq`i' = compwptfaeq`i' + compwptfleq`i' 

		gen compwptfadeb`i' = (rpdeb_a`i'*diff_ratio_ptfxa_deb`i'*ptdxa`i')/gdpro`i' 
		gen compwptfldeb`i' = -(rpdeb_l`i'*diff_ratio_ptfxd_deb`i')*ptdxd`i'/gdpro`i'
		gen compwptfdeb`i' = compwptfadeb`i' + compwptfldeb`i' 
	
		gen compwptfares`i' = (rpres_a`i'*diff_ratio_ptfxa_res`i')*ptrxa`i'/gdpro`i' 

	gen compwfdia`i' = (rf_a`i'*diff_ratio_fdixa`i')*fdixa`i'/gdpro`i' 
	gen compwfdil`i' = - (rf_l`i'*diff_ratio_fdixd`i')*fdixd`i'/gdpro`i' 
	gen compwfdi`i' = compwfdia`i' + compwfdil`i'

}

keep period priv* ratio* r_a* r_l* rf_a* rf_l* rp* nwgxa* nwgxd* fdixa* fdixd* ptfxa* ptfxd* ptexa* ptdxa* ptrxa* ptexd* ptdxd* gdpro* comp*
reshape long r_a r_l rf_a rf_l rp_a rp_l rpeq_a rpeq_l rpdeb_a rpdeb_l rpres_a gdpro priv privw privwptf privwfdi ratio_ptfxa ratio_ptfxa_eq ratio_ptfxa_deb ratio_ptfxa_res ratio_fdixa ratio_ptfxd ratio_ptfxd_eq ratio_ptfxd_deb ratio_fdixd nwgxa nwgxd fdixa fdixd ptfxa ptfxd privwa privwl privwptfa privwptfl privwfdia privwfdil privwptfeq privwptfaeq privwptfleq privwptfdeb privwptfadeb privwptfldeb privwptfares compwa compwl compw compwptfa compwptfl compwptf compwptfaeq compwptfleq compwptfeq compwptfadeb compwptfldeb compwptfdeb compwptfares compwfdia compwfdil compwfdi, i(period) j(qanninc) 
order period, after(qanninc)
so qanninc period 
label values qanninc drop  
order qanninc period r_a r_l rf_a rf_l rp_a rp_l rpeq_a rpeq_l rpdeb_a rpdeb_l rpres_a ratio_ptfxa ratio_ptfxa_eq ratio_ptfxa_deb ratio_ptfxa_res ratio_fdixa ratio_ptfxd ratio_ptfxd_eq ratio_ptfxd_deb ratio_fdixd priv nwgxa nwgxd fdixa fdixd ptfxa ptfxd gdpro privw privwa privwl privwptf privwptfa privwptfl privwptfeq privwptfaeq privwptfleq privwptfdeb privwptfadeb privwptfldeb privwptfares privwfdi privwfdia privwfdil compwa compwl compw compwptfa compwptfl compwptf compwptfaeq compwptfleq compwptfeq compwptfadeb compwptfldeb compwptfdeb compwptfares compwfdia compwfdil compwfdi 

// export excel "$graph/quintiles/returns/decomp_diff_world_averages.xlsx",  firstrow(var) sheet("composition_return_tables", replace)

// graphs 
keep qanninc period ratio_ptfxa_eq ratio_ptfxa_deb ratio_ptfxa_res ratio_fdixa ratio_ptfxd_eq ratio_ptfxd_deb ratio_fdixd
reshape long ratio_, i(qanninc period) j(type) string

gen class = "Assets" if inlist(type, "fdixa", "ptfxa_eq", "ptfxa_res", "ptfxa_deb")
replace class = "Liabilities" if mi(class)
replace class = "A" if class == "Assets" 
replace class = "L" if class == "Liabilities"
replace type = "Equity" if inlist(type, "ptfxa_eq", "ptfxd_eq")
replace type = "Debt" if inlist(type, "ptfxa_deb", "ptfxd_deb")
replace type = "Reserves" if inlist(type, "ptfxa_res")
replace type = "FDI" if inlist(type, "fdixa", "fdixd")

label values qanninc Qanninc
label def Qanninc 1 "Bottom 20%", modify
label def Qanninc 2 "20-40%", modify
label def Qanninc 3 "40-60%", modify
label def Qanninc 4 "60-80%", modify
label def Qanninc 5 "Top 20%", modify

gen quint = "Bottom 20%" if qanninc == 1
replace quint = "20-40%" if qanninc == 2
replace quint = "40-60%" if qanninc == 3
replace quint = "60-80%" if qanninc == 4
replace quint = "Top 20%" if qanninc == 5

so quint period



mylabels 0(10)100, myscale(@/100) local(myla) suffix("%")

graph bar ratio_ if period == "1970-1999" & qanninc != 99, ///
    over(type) over(class, gap(*.1) label(labsize(huge))) ///
    over(qanninc, label(labsize(huge))) ylabel(`myla', labsize(huge)) ///
    asyvars stack ytitle("") legend(rows(1)) ///
    bar(1, color("0 104 180"))   /// Debt = #0068B4
    bar(2, color("227 0 11"))    /// Equity = #E3000B
    bar(3, color("44 160 44"))   /// FDI = #2CA02C
    bar(4, color("255 221 0"))   /// Reserves = #FFDD00
    subtitle("1970-1999", size(huge))
graph save "$outputs/images/decomp_NFA_period1.gph", replace

graph bar ratio_ if period == "2000-2024" & qanninc != 99, ///
    over(type) over(class, gap(*.1) label(labsize(huge))) ///
    over(qanninc, label(labsize(huge))) ylabel(`myla', labsize(huge)) ///
    asyvars stack ytitle("") legend(rows(1)) ///
    bar(1, color("0 104 180"))   /// Debt = #0068B4
    bar(2, color("227 0 11"))    /// Equity = #E3000B
    bar(3, color("44 160 44"))   /// FDI = #2CA02C
    bar(4, color("255 221 0"))   /// Reserves = #FFDD00
    subtitle("2000-2024", size(huge))
graph save "$outputs/images/decomp_NFA_period2.gph", replace

* Paragraph-style caption that auto-wraps
local cap ///
"{bf:Interpretation.} This figure shows the decomposition of foreign assets (A) and liabilities (L) across country income groups in two periods: 1970-1999 and " ///
"2000-2024. Rich countries (top 20%) hold fewer foreign exchange reserves and issue fewer Foreign Direct Investment (FDI) liabilities, both of which are low-" ///
"return components. Instead, they have increased their share of equity and FDI assets, which typically yield higher returns. On the liability side, they continue " ///
"to rely on debt issuance, which is safer and lower-cost due to their strong credit ratings and reserve currency status. In contrast, poorer countries (bottom " ///
"80%) have portfolios skewed toward reserves as assets and FDI as liabilities, both associated with lower net returns. Their debt liabilities remain significant, " ///
"often at higher interest rates, reflecting weaker sovereign ratings. {bf:Sources and series:} Nievas and Sodano (2024) and wir2026.wid.world/methodology."



grc1leg "$outputs/images/decomp_NFA_period1.gph" "$outputs/images/decomp_NFA_period2.gph", ///
    subtitle("Decomposition of Assets (A) and Liabilities (L)") ///
    ycommon legendfrom("$outputs/images/decomp_NFA_period1.gph") ///
    graphregion(fcolor(white) margin(l+4 r+4 t+6 b+22)) ///
    scale(0.8) iscale(*.6) ///
    caption("`cap'", span justification(left) size(small))

graph export "$outputs\Graphs\Ch5/decomposition_NFAperiods.png", replace width(3000) height(1800)
graph export "$outputs\Graphs\Ch5/decomposition_NFAperiods.pdf", replace 


export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-F5.6") sheetmod cell(A2) first(varl)

 putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-F5.6")
putexcel (B2:K60), nformat(percent)
 
 
 



**# Figure 5.5.b Rich countries have become global financial rentiers by design, not by accident
/*

 (Nievas and Sodano (2025))  
*/
//******************************************************************************
// Graphs 
//******************************************************************************
// import excel "$raw/country-codes-core.xlsx", clear firstrow
// ren code iso
// g year =2024
// save "$work_data/country-codes-list-core-year.dta",replace

u "$raw/currency-composition.dta", clear
drop geo* 

merge 1:1 iso year using "$work_data//wid-database_privilege.dta", nogen keepusing(gdpro nyixx lcusx nwgxa nwgxd)


foreach var in nwgxa nwgxd  {
	gen `var'_idx = `var'*nyixx
	replace `var' = `var'_idx/lcusx
}

// 	merge 1:1 iso year using "$work_data/country-codes-list-core-year.dta", nogen keepusing(corecountry country TH) 
// 	keep if corecountry == 1

g corecountry=.
foreach country in $corecountries{
	replace corecountry=1 if iso=="`country'"
}
keep if corecountry==1

drop if year < 1990	

sort iso year 
carryforward w_a_dc w_a_usd w_a_eur w_a_gbp w_a_jpy w_a_cny w_a_oth w_l_dc w_l_usd w_l_eur w_l_gbp w_l_jpy w_l_cny w_l_oth w_a_chf w_l_chf if year > 2017, replace 

// regional average for missing countries
foreach level in undet un {
	kountry iso, from(iso2c) geo(`level')

replace GEO = "Western Asia" 	if iso == "AE" & "`level'" == "undet"
replace GEO = "Caribbean" 		if iso == "CW" & "`level'" == "undet"
replace GEO = "Caribbean"		if iso == "SX" & "`level'" == "undet"
replace GEO = "Caribbean" 		if iso == "BQ" & "`level'" == "undet"
replace GEO = "Southern Europe" if iso == "KS" & "`level'" == "undet"
replace GEO = "Southern Europe" if iso == "ME" & "`level'" == "undet"
replace GEO = "Eastern Asia" 	if iso == "TW" & "`level'" == "undet"
replace GEO = "Northern Europe" if iso == "GG" & "`level'" == "undet"
replace GEO = "Northern Europe" if iso == "JE" & "`level'" == "undet"
replace GEO = "Northern Europe" if iso == "IM" & "`level'" == "undet"

replace GEO = "Asia" if inlist(iso, "AE", "TW") & "`level'" == "un"
replace GEO = "Americas" if inlist(iso, "CW", "SX", "BQ") & "`level'" == "un"
replace GEO = "Europe" if inlist(iso, "KS", "ME", "GG", "JE", "IM") & "`level'" == "un"
ren GEO geo`level'
drop NAMES_STD 
}

foreach v in w_a_dc w_a_usd w_a_eur w_a_gbp w_a_jpy w_a_cny w_a_oth w_l_dc w_l_usd w_l_eur w_l_gbp w_l_jpy w_l_cny w_l_oth w_a_chf w_l_chf { 
	
 foreach level in undet un {
		
  bys geo`level' year : egen av`level'`v' = mean(`v')

  }

replace `v' = avundet`v' if missing(`v') 
replace `v' = avun`v' if missing(`v')
*replace `v' = avworld`v' if missing(`v') & auxexp == 0 
}

gen euro = .
replace euro = 1 if iso == "BE" & year < 1999
replace euro = 1 if iso == "DE" & year < 1999
replace euro = 1 if iso == "ES" & year < 1999
replace euro = 1 if iso == "FR" & year < 1999
replace euro = 1 if iso == "IE" & year < 1999
replace euro = 1 if iso == "IT" & year < 1999
replace euro = 1 if iso == "LU" & year < 1999
replace euro = 1 if iso == "NL" & year < 1999
replace euro = 1 if iso == "AT" & year < 1999
replace euro = 1 if iso == "PT" & year < 1999
replace euro = 1 if iso == "FI" & year < 1999
*replace euro = 1 if iso == "GR" & year < 2001
*replace euro = 1 if iso == "SI" & year < 2007
replace euro = 1 if iso == "CY" & year < 1999
*replace euro = 1 if iso == "MT" & year < 2008
*replace euro = 1 if iso == "SK" & year < 2009
*replace euro = 1 if iso == "EE" & year < 2011
*replace euro = 1 if iso == "LV" & year < 2014
*replace euro = 1 if iso == "LT" & year < 2015

replace w_a_eur = w_a_dc if euro == 1 
replace w_a_dc = 0 if euro == 1 
replace w_l_eur = w_l_dc if euro == 1 
replace w_l_dc = 0 if euro == 1

// reschaling for these new shares
egen checkassets = rowtotal(w_a_dc w_a_usd w_a_eur w_a_gbp w_a_jpy w_a_cny w_a_oth w_a_chf)
foreach var in w_a_dc w_a_usd w_a_eur w_a_gbp w_a_jpy w_a_cny w_a_oth w_a_chf {
	replace `var' = `var'/checkassets
}

egen checkliab = rowtotal(w_l_dc w_l_usd w_l_eur w_l_gbp w_l_jpy w_l_cny w_l_oth w_l_chf)
foreach var in w_l_dc w_l_usd w_l_eur w_l_gbp w_l_jpy w_l_cny w_l_oth w_l_chf {
	replace `var' = `var'/checkliab
}
drop check*


// getting absolute values 
foreach var in w_a_dc w_a_usd w_a_eur w_a_gbp w_a_jpy w_a_cny w_a_oth w_a_chf {
	replace `var' = `var'*nwgxa
}

foreach var in w_l_dc w_l_usd w_l_eur w_l_gbp w_l_jpy w_l_cny w_l_oth w_l_chf {
	replace `var' = `var'*nwgxd
}

replace nwgxa =. if mi(w_a_dc) & mi(w_a_usd) & mi(w_a_eur) & mi(w_a_gbp) & mi(w_a_jpy) & mi(w_a_cny) & mi(w_a_cny) & mi(w_a_oth)
replace nwgxd =. if mi(w_l_dc) & mi(w_l_usd) & mi(w_l_eur) & mi(w_l_gbp) & mi(w_l_jpy) & mi(w_l_cny) & mi(w_l_chf) & mi(w_l_oth)

collapse (sum) w_a_dc w_a_usd w_a_eur w_a_gbp w_a_jpy w_a_cny w_a_oth w_a_chf nwgxa w_l_dc w_l_usd w_l_eur w_l_gbp w_l_jpy w_l_cny w_l_oth w_l_chf nwgxd, by(year)

foreach var in w_a_dc w_a_usd w_a_eur w_a_gbp w_a_jpy w_a_cny w_a_oth w_a_chf {
	replace `var' = `var'/nwgxa
}

replace w_a_oth = w_a_oth + w_a_dc

mylabels 0(10)50, myscale(@/100) local(myla) suffix("%")
twoway (line w_a_usd year, lwidth(medthick) lcolor("64 105 166"))			 	 ///
	   (line w_a_eur year, lwidth(medthick) lcolor("200 0 0"))			 	 ///	   
	   (line w_a_gbp year, lwidth(medthick) lcolor("146 195 51"))			 	 ///
	   (line w_a_cny year, lwidth(medthick) lcolor("orange"))			 		 ///
	   (line w_a_jpy year, lwidth(medthick) lcolor("246 199 0"))			  		 ///
	   (line w_a_chf year, lwidth(medthick) lcolor("cranberry*.55"))         ///
	   (line w_a_oth year, lwidth(medthick) lcolor("green"))                         ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(1 "US Dollar" 2 "Euro" 3 "Pound sterling" 4 "Chinese renminbi" 5 "Japanese yen" 6 "Swiss franc" 7 "Other currencies") rows(2) pos(6)) note("") ylabel(, grid glcolor(gs15)) xlabel(1990(4)2022, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3)) yla(`myla', ang(h))

keep year w_a_usd w_a_eur w_a_gbp w_a_cny w_a_jpy w_a_chf w_a_oth 
label var w_a_usd "US Dollar"
label var w_a_eur "Euro"
label var w_a_gbp "Pound sterling"
label var w_a_cny "Chinese renminbi"
label var w_a_jpy "Japanese yen"
label var w_a_chf "Swiss franc"
label var w_a_oth "Other currencies"
export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-Figure5.5b") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-Figure5.5b")
putexcel (B3:H38), nformat(percent)
// graph export "$graph/currency_nwgxa.pdf", replace 





**# Figure 5.5.c Rich countries have become global financial rentiers by design, not by accident
/*

 (Nievas and Sodano (2025))  
*/
//******************************************************************************
// Graphs 
//******************************************************************************

u "$raw//reserves_1976_2022.dta", clear 

*foreach v in  w_chf w_cny w_eur  w_gbp w_jpy  w_oth w_usd {
*replace `v'= `v'*100
*}

*niceloglabels 0.01  100, local(yla) style(1) 
mylabels 0(10)80, myscale(@/100) local(myla) suffix("%")
twoway (line w_usd year if country == "World", lwidth(medthick) lcolor("64 105 166"))			 	 ///
	   (line w_eur year if country == "World", lwidth(medthick) lcolor("200 0 0"))			 	 ///	   
	   (line w_gbp year if country == "World", lwidth(medthick) lcolor("146 195 51"))			 	 ///
	   (line w_cny year if country == "World", lwidth(medthick) lcolor("orange"))			 		 ///
	   (line w_jpy year if country == "World", lwidth(medthick) lcolor("246 199 0"))			  		 ///
	   (line w_chf year if country == "World", lwidth(medthick) lcolor("cranberry*.55"))         ///
	   (line w_oth year if country == "World", lwidth(medthick) lcolor("green"))                         ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(1 "US Dollar" 2 "Euro" 3 "Pound sterling" 4 "Chinese renminbi" 5 "Japanese yen" 6 "Swiss franc" 7 "Other currencies") rows(2) pos(6)) note("") ylabel(`myla', grid glcolor(gs15)) xlabel(1976(4)2022, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3)) 

lab var w_usd "US Dollar"
lab var w_eur "Euro"
lab var w_gbp "Pound sterling"
lab var w_cny "Chinese renminbi"
lab var w_jpy "Japanese yen"
lab var w_chf "Swiss franc"
lab var w_oth "Other currencies"

keep if country=="World"
drop country tot

export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-Figure5.5c") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-Figure5.5c")
putexcel (B3:H49), nformat(percent)





**# Figure 5.5.a Rich countries have become global financial rentiers by design, not by accident
/*

 (Nievas and Sodano (2025))  
*/
//******************************************************************************
// Data from Boz et al. 
//******************************************************************************
import excel "$raw\all countries.xlsx", clear firstrow sheet("Annual")

drop if mi(ISO3C)

kountry ISO3C, from(iso3c) to(iso2c)
tab ISO3C if mi(_ISO2C_)
ren _ISO2C_ iso

egen ExportOth = rowtotal(ExportOther_Excl_USDEUR ExportUnclassified), missing
egen ImportOth = rowtotal(ImportOther_Excl_USDEUR ImportUnclassified), missing

keep iso year ExportUSD ExportEUR ExportOth ImportUSD ImportEUR ImportOth 
order iso year ExportUSD ExportEUR ExportOth ImportUSD ImportEUR ImportOth 

renvars, lower

sort iso year 
carryforward exportusd exporteur exportoth importusd importeur importoth if year == 2020, replace 

foreach var in exportusd exporteur exportoth importusd importeur importoth { 
	replace `var' = `var'/100
}

// other exports
drop exportoth importoth 
gen exportoth = 1 - cond(!mi(exportusd), exportusd, 0) - cond(!mi(exporteur), exporteur, 0)
replace exportoth = . if mi(exportusd) & mi(exporteur)

// other imports
gen importoth = 1 - cond(!mi(importusd), importusd, 0) - cond(!mi(importeur), importeur, 0)
replace importoth = . if mi(importusd) & mi(importeur)

// merge with tradebalances 
merge 1:1 iso year using "$raw/tradebalances.dta", nogen keepusing(tradebalance exports imports)
drop if year < 1990 


// regional average for missing countries
foreach level in undet un {
	kountry iso, from(iso2c) geo(`level')

replace GEO = "Western Asia" 	if iso == "AE" & "`level'" == "undet"
replace GEO = "Caribbean" 		if iso == "CW" & "`level'" == "undet"
replace GEO = "Caribbean"		if iso == "SX" & "`level'" == "undet"
replace GEO = "Caribbean" 		if iso == "BQ" & "`level'" == "undet"
replace GEO = "Southern Europe" if iso == "KS" & "`level'" == "undet"
replace GEO = "Southern Europe" if iso == "ME" & "`level'" == "undet"
replace GEO = "Eastern Asia" 	if iso == "TW" & "`level'" == "undet"
replace GEO = "Northern Europe" if iso == "GG" & "`level'" == "undet"
replace GEO = "Northern Europe" if iso == "JE" & "`level'" == "undet"
replace GEO = "Northern Europe" if iso == "IM" & "`level'" == "undet"

replace GEO = "Asia" if inlist(iso, "AE", "TW") & "`level'" == "un"
replace GEO = "Americas" if inlist(iso, "CW", "SX", "BQ") & "`level'" == "un"
replace GEO = "Europe" if inlist(iso, "KS", "ME", "GG", "JE", "IM") & "`level'" == "un"
ren GEO geo`level'
drop NAMES_STD 
}

egen auxexp = rowtotal(exportusd exporteur exportoth)
egen auximp = rowtotal(importusd importeur importoth)

gen auxexpeur = exporteur 
replace exporteur =. if year <= 1999
foreach v in exportusd exporteur exportoth { 
	
 foreach level in undet un {
		
  bys geo`level' year : egen av`level'`v' = mean(`v')

  }
  bys year : egen avworld`v' = mean(`v') 

replace `v' = avundet`v' if missing(`v') & auxexp == 0
replace `v' = avun`v' if missing(`v') & auxexp == 0 
*replace `v' = avworld`v' if missing(`v') & auxexp == 0 
}
replace exporteur = auxexpeur if !mi(auxexpeur) & year <= 1999

gen auximpeur = importeur 
replace importeur =. if year <= 1999
foreach v in importusd importeur importoth { 
	
 foreach level in undet un {
		
  bys geo`level' year : egen av`level'`v' = mean(`v')

  }
  bys year : egen avworld`v' = mean(`v') 

replace `v' = avundet`v' if missing(`v') & auximp == 0
replace `v' = avun`v' if missing(`v') & auximp == 0 
*replace `v' = avworld`v' if missing(`v') & auximp == 0 
}
replace importeur = auximpeur if !mi(auximpeur) & year <= 1999
drop aux* av* geo* 


// reschaling for these new shares
egen checkexp = rowtotal(exportusd exporteur exportoth)
foreach var in exportusd exporteur exportoth {
	replace `var' = `var'/checkexp
}

egen checkimp = rowtotal(importusd importeur importoth)
foreach var in importusd importeur importoth {
	replace `var' = `var'/checkimp
}
drop check*

// getting absolute values 
foreach var in exportusd exporteur exportoth {
	replace `var' = `var'*exports
}

foreach var in importusd importeur importoth {
	replace `var' = `var'*imports
}

// dropping not to inflate totals 
replace exports =. if mi(exportusd) & mi(exporteur) & mi(exportoth)
replace imports =. if mi(importusd) & mi(importeur) & mi(importoth)

collapse (sum) exportusd exporteur exportoth exports importusd importeur importoth imports, by(year)

// again shares
foreach var in exportusd exporteur exportoth {
	replace `var' = `var'/exports
}

foreach var in importusd importeur importoth {
	replace `var' = `var'/imports
}


//Graphs 
drop if year > 2020
// Gross foreign assets (corrected for offshore wealth) as share of GDP per income group 
mylabels 0(10)80, myscale(@/100) local(myla) suffix("%")
twoway (line importusd year, lwidth(medthick) lcolor("64 105 166"))			 	 ///
	   (line importeur year, lwidth(medthick) lcolor("200 0 0"))			 	 ///	   
	   (line importoth year, lwidth(medthick) lcolor("green"))                   ///
, title("") subtitle("") ytitle("") xtitle("") legend(order(1 "US Dollar" 2 "Euro" 3 "Other") rows(1)) note("") ylabel(, grid glcolor(gs15)) xlabel(1990(5)2020, grid glcolor(gs15)) scale(0.8) graphregion(color(white)) graphregion(margin(l+0 r+3)) yla(`myla', ang(h))

label var importusd "Us Dollar"
label var importeur "Euro"
label var importoth "Other currencies"
keep year  importusd importeur importoth



export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-Figure5.5a") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-Figure5.5a")
putexcel (B3:H49), nformat(percent)







**# Figure 5.8. Poorer countries can spend less on public services, exacerbating inequality
/*

 (Nievas and Sodano (2025))  
*/
//******************************************************************************
// Privilege vs Human capital
//******************************************************************************
// Import figures on education and health public investment

use if widcode == "mheage999i" | widcode=="meduge999i" using $wid2024, clear
drop p currency 
reshape wide value, i(iso year) j(widcode) string
renvars value*, predrop(6)
renvars *999i, postdrop(4)


g corecountry=.
foreach country in $corecountries{
	replace corecountry=1 if iso=="`country'"
}
keep if corecountry==1

save "$work_data/humancapital-wid.dta", replace
	
// merging with rest of data
u "$work_data/humancapital-wid.dta", clear
merge 1:1 iso year using "$work_data//wid-database_privilege.dta", keepusing(gdpro nyixx lcusx nwgxa nwgxd pinrx pinpx) nogen

foreach var in gdpro nwgxa nwgxd pinrx pinpx heage eduge {
	gen `var'_idx = `var'*nyixx
	replace `var' = `var'_idx/lcusx
}


gen nwnxa = nwgxa - nwgxd

// wealth stock is the one from the previous period
encode iso, gen(i)
xtset i year 
foreach var in nwgxa nwnxa nwgxd {
	gen orig`var' = `var'
	gen aux`var' = l.`var'
	replace `var' = aux`var'
}
drop i aux*

// return rates
gen r_a = pinrx/nwgxa
gen r_l = pinpx/nwgxd

// excess yields
gen priv = r_a - r_l

// Excess yields as a share of GDP
gen exc =  priv*nwgxa if priv > 0
replace exc = priv*nwgxd if priv < 0
replace exc =  exc/gdpro

foreach var in heage eduge {
	gen `var'_gdp = `var'/gdpro 
}

*quintiles
merge 1:m iso year using "$work_data\quintiles", nogen


foreach var in gdpro nwgxa nwgxd pinrx pinpx heage eduge {

replace `var' = `var'*weight/popul

} 

collapse (sum) gdpro nwgxa nwgxd pinrx pinpx heage eduge, by(qanninc year)

// return rates
gen r_a = pinrx/nwgxa
gen r_l = pinpx/nwgxd

// excess yields
gen priv = r_a - r_l

// Excess yields as a share of GDP
gen exc =  priv*nwgxa if priv > 0
replace exc = priv*nwgxd if priv < 0
replace exc =  exc/gdpro

foreach var in heage eduge {
	gen `var'_gdp = `var'/gdpro 
}

keep if year == 2022 

replace exc = abs(exc)

gen quint = ""
replace quint = "Bottom 20%" if qanninc == 1 
replace quint = "20%-40%" if qanninc == 2 
replace quint = "40%-60%" if qanninc == 3 
replace quint = "60%-80%" if qanninc == 4 

drop if qanninc == 5 

label define qanninc_lbl 1 "Bottom 20%" 2 "20%-40%" 3 "40%-60%" 4 "60%-80%" 5 "Top 20%"
label values qanninc qanninc_lbl

keep year exc heage_gdp eduge_gdp qanninc

mylabels 0(1)5, myscale(@/100) local(myla) suffix("%")
graph bar exc heage_gdp eduge_gdp, over(qanninc, gap(50)) bargap(20) ///
    legend(label(1 "Privilege") label(2 "Health") label(3 "Education") rows(1) pos(6)) ///
    ytitle("% of GDP") title("") ylabel(`myla', grid glcolor(gs15))
	
label var exc "Privilege"
label var heage_gdp "Health"
label var eduge_gdp	"Education"
	
// graph export "$graph/quintiles/humancapital.pdf", replace 

export excel using "$outputs/WIR_2026_Chapter5.xlsx", sheet("data-Figure5.8") sheetmod cell(A2) first(varl)
putexcel set "$outputs/WIR_2026_Chapter5.xlsx", modify sheet("data-Figure5.8")
putexcel (B2:H49), nformat(percent)



**# Figure 5.9. These structural asymmetries call for reforms in the international financial, trade, and monetary system
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Nievas and Piketty (2025)
Figures 15 (c), 22 (d), 23 (a), and 25 (b)  in Nievas and Piketty (2025)

Paper
https://wid.world/document/unequal-exchange-and-north-south-relations-evidence-from-global-trade-flows-and-the-world-balance-of-payments-1800-2025-world-inequality-lab-working-paper-2025-11/

Replication package
https://wid.world/document/data-file-to-unequal-exchange-and-north-south-relations-evidence-from-global-trade-flows-and-the-world-balance-of-payments-1800-2025-world-inequality-lab-working-paper-2025-11/
*/


**# Figure Box 5.1. Exorbitant duty is not so exorbitant
* ---------------------------------------------------------------------------- *
* ---------------------------------------------------------------------------- * 
/*
Data from Nievas and Sodano (2025)
Table 45 in Nievas and Sodano (2025)

Paper
https://wid.world/document/has-the-us-exorbitant-privilege-become-a-rich-world-privilege-rates-of-return-and-foreign-assets-from-a-global-perspective-1970-2022-world-inequality-lab-working-paper-2024-14/

*/
